;; -*- mode: scheme; coding: utf-8 -*-
;; Copyright (c) 2025 Vadym Kochan
;; SPDX-License-Identifier: MIT
#!r6rs

(library (conbot)
  (export conbot-default-timeout
          conbot-default-cols
          conbot-default-rows

          text-row?
          text-row-string
          text-row-length
          text-row-prev
          text-row-next

          open-conbot-process
          conbot-close
          conbot?
          conbot-prompt
          conbot-prompt-set!
          conbot-timeout
          conbot-timeout-set!
          conbot-command
          conbot-running?
          conbot-exit-signal
          conbot-exit-code
          conbot-wait-exit
          conbot-wait-match
          conbot-wait-prompt
          conbot-do-shell
          conbot-shell-eol
          conbot-shell-eol-set!
          conbot-write-char
          conbot-write-key
          conbot-write
          conbot-clear
          conbot-update

          conbot-fetch-lines
          conbot-fetch-lines-by-prompt

          conbot-screen-string
          conbot-screen-ref
          conbot-rows-ref
          conbot-cursor-row
          conbot-first-row
          conbot-last-row
          conbot-end-row

          conbot-match?
          conbot-match-start
          conbot-match-end
          conbot-match-substring)
  (import (rnrs)
          (rnrs mutable-strings)
          (srfi :19)
          (srfi :39)
          (vterm)
	  (vterm common)
	  (vterm events)
	  (vterm screen)
          (wak irregex)
          (conbot private io)
          (conbot private pty))

(define conbot-default-timeout (make-parameter 60000))
(define conbot-default-cols (make-parameter 80))
(define conbot-default-rows (make-parameter 25))

(define-record-type text-row
  (fields (mutable prev)
          (mutable next)
          (mutable raw-string)
          (mutable cached)
          (mutable cr)
          (mutable nl))
  (protocol
    (lambda (new)
      (lambda (prev next str)
        (new prev next str #f #f #f)))))

(define (text-row-string row)
  (or (text-row-cached row)
      (cond ((text-row-nl row) =>
             (lambda (nl)
               (let ([s (text-row-raw-string row)])
                 (let ([c (string-append (substring s 0 nl) "\n")])
                   (text-row-cached-set! row c)
                   c))))
            (else (text-row-raw-string row)))))

(define (text-row-length row)
  (string-length (text-row-string row)))

(define (text-row-remove row)
  (assert row)
  (let ([prev (text-row-prev row)]
        [next (text-row-next row)])
    (when prev
      (text-row-next-set! prev next))
    (when next
      (text-row-prev-set! next prev))
    (text-row-prev-set! row #f)
    (text-row-next-set! row #f)))

(define (text-row-insert-after row after)
  (assert row)
  (assert after)
  (let ([next (text-row-next after)])
    (when next
      (text-row-prev-set! next row))
    (text-row-next-set! after row)
    (text-row-prev-set! row after)
    (text-row-next-set! row next)))

(define (text-row-insert-before row before)
  (assert row)
  (assert before)
  (let ([prev (text-row-prev before)])
    (when prev 
      (text-row-next-set! prev row))
    (text-row-prev-set! before row)
    (text-row-prev-set! row prev)
    (text-row-next-set! row before)))

(define (make-conbot-chunker)
  (define (get-next row)
    (let ([next (text-row-next row)])
      (and next (text-row-string next) next)))

  (define (get-string row)
    (text-row-string row))

  (make-irregex-chunker get-next get-string))

(define-record-type conbot
  (fields (immutable command)
          (mutable process)
          (mutable has-data?)
          (immutable rows-head)
          (immutable rows-tail)
          (mutable top-row)
          (mutable prompt-row)
          (mutable prompt-pos)
          (immutable chunker)
          (mutable prompt)
          (mutable timeout)
          (mutable shell-eol)
          (immutable vterm))
  (protocol
    (lambda (new)
      (lambda (cmd)
        (letrec ([cb (new cmd ;; command
                          #f  ;; process
                          #f  ;; has-data?
                          (make-text-row #f #f #f) ;; rows-head
                          (make-text-row #f #f #f) ;; rows-tail
                          #f ;; top-row
                          #f ;; prompt-row
                          0 ;; prompt-pos
                          (make-conbot-chunker) ;; chunker
                          #f ;; prompt
                          (conbot-default-timeout) ;; timeout
                          "\n" ;; shell-eol
                          (make-vterm/screen (conbot-default-rows) (conbot-default-cols) ;; vterm
                            (lambda (obj evt)
                              (cond
                                ((vterm-control-event? evt)
                                 (cond ((eqv? #xd (vterm-control-event-code evt))
                                        (let ([row (conbot-screen-ref cb (vterm-cursor-row (conbot-vterm cb)))])
                                          (text-row-cr-set! row (vterm-cursor-col (conbot-vterm cb)))))

                                       ((eqv? #xa (vterm-control-event-code evt))
                                        (let* ([row (conbot-screen-ref cb (vterm-cursor-row (conbot-vterm cb)))]
                                               [col (or (text-row-cr row) (vterm-cursor-col (conbot-vterm cb)))]
                                               [str (text-row-string row)])
                                          (text-row-nl-set! row col)))))

                                ((vterm-push-line-event? evt)
                                 (conbot-push-row cb))

                                ((vterm-damage-event? evt)
                                 (conbot-has-data?-set! cb #t)
                                 (vterm-for-each-cell (conbot-vterm cb) (vterm-damage-event-rect evt)
                                   (lambda (cell row col)
                                     (if (eqv? (vterm-cell-char cell) 'null)
                                         (conbot-put-char cb col row #\space)
                                         (conbot-put-char cb col row (vterm-cell-char cell)))))
                                ))
                            
                            )))])
            (text-row-next-set! (conbot-rows-head cb) (conbot-rows-tail cb))
            (text-row-prev-set! (conbot-rows-tail cb) (conbot-rows-head cb))
            (do ([i 0 (fx+ i 1)])
                ((fx=? i (conbot-default-rows)))
              (let ([row (make-text-row #f #f (make-string (conbot-default-cols) #\space))])
                (text-row-insert-before row (conbot-rows-tail cb))))
            (conbot-top-row-set! cb (conbot-first-row cb))
            (vterm-scroll-buffer-enable! (conbot-vterm cb) #t)
          cb)))))

(define open-conbot-process
  (case-lambda
    [(cmd)
     (open-conbot-process cmd #f) ]

    [(cmd env)
     (let ([cb (make-conbot cmd)])
       (conbot-spawn cb env)
       cb)] ))

(define (conbot-push-row cb)
  (let ([new-top (text-row-next (conbot-top-row cb))])
    (conbot-top-row-set! cb new-top))
  (let ([new-row (make-text-row #f #f (make-string (vterm-cols (conbot-vterm cb)) #\space))])
    (text-row-insert-after new-row (conbot-last-row cb))))

(define (conbot-screen-string cb)
  (vterm-string (conbot-vterm cb)))

(define (conbot-screen-ref cb row)
  (let loop ([i 0] [r (conbot-top-row cb)])
    (cond ((fx<? i row)
           (loop (fx+ i 1) (text-row-next r)))
          (else r))))

(define (conbot-put-char cb col row ch)
  (let* ([row (conbot-screen-ref cb row)]
         [str (text-row-raw-string row)])
    (text-row-cached-set! row #f)
    (string-set! str col ch)))

(define (conbot-cursor-row cb)
  (conbot-screen-ref cb (vterm-cursor-row (conbot-vterm cb))))

(define (conbot-rows-ref cb row)
  (let loop ([i 0] [r (conbot-first-row cb)])
    (cond ((fx<? i row)
           (loop (fx+ i 1) (text-row-next r)))
          (else r))))

(define (conbot-row-num cb row)
  (let loop ([i 0] [r (conbot-first-row cb)])
    (cond ((not (eqv? r row))
           (loop (fx+ i 1) (text-row-next r)))
          (else i))))

(define (conbot-first-row cb)
  (text-row-next (conbot-rows-head cb)))

(define (conbot-last-row cb)
  (text-row-prev (conbot-rows-tail cb)))

(define (conbot-end-row cb)
  (conbot-rows-tail cb))

(define-record-type conbot-match
  (fields regex))

(define (conbot-match-start cm)
  (irregex-match-start-index (conbot-match-regex cm) 0))

(define (conbot-match-end cm)
  (irregex-match-end-index (conbot-match-regex cm) 0))

(define (conbot-match-substring cm)
  (irregex-match-substring (conbot-match-regex cm) 0))

(define (conbot-spawn cb env)
  (let ([cmd (conbot-command cb)])
    (conbot-process-set! cb (open-pty-process "/bin/sh"
                                              (and cmd (list "-c" cmd)) env))
    (vterm-output-port-set! (conbot-vterm cb) (process-pty-port (conbot-process cb))) ))

(define (conbot-close cb)
  (process-kill (conbot-process cb))
  (conbot-wait-exit cb))

(define (conbot-running? cb)
  (and (conbot-process cb) (process-alive? (conbot-process cb))))

(define (conbot-read-timeout cb timeout)
  (let ([state (io-poll-fd (process-pty-fd (conbot-process cb)) timeout)])
    (when (eqv? state 'ready)
      (let ([bv (get-bytevector-some (process-pty-port (conbot-process cb)))])
        (and (bytevector? bv)
             (vterm-process (conbot-vterm cb) bv))))))

(define conbot-read
  (case-lambda
    [(cb)
     (conbot-read-timeout cb (conbot-default-timeout)) ]

    [(cb timeout)
     (conbot-read-timeout cb timeout) ]))

(define (conbot-update cb)
  (conbot-read cb))

(define (conbot-exit-signal cb)
  (process-exit-signal (conbot-process cb)))

(define (conbot-exit-code cb)
  (process-exit-code (conbot-process cb)))

(define conbot-wait-exit
  (case-lambda
    [(cb)
     (process-wait (conbot-process cb)) ]

    [(cb timeout)
     (process-wait (conbot-process cb) timeout) ]))

(define (time-millisecond tm)
  (fx+ (fx* (time-second tm) 1000)
       (fxdiv (time-nanosecond tm) 1000000)))

(define (conbot-search cb regx from-row start-pos)
  (let ([m (irregex-search/chunked (irregex regx 'm) (conbot-chunker cb) from-row start-pos)])
    (and m (make-conbot-match m))))

(define conbot-wait-match
  (case-lambda
    [(cb regx)
     (conbot-wait-match cb regx (conbot-first-row cb) 0) ]

    [(cb regx from-row)
     (conbot-wait-match cb regx from-row 0) ]

    [(cb regx from-row start-pos)
     (let ([start-time (current-time)])
       (let loop ([diff (time-difference (current-time) start-time)])
         (cond
           ((and (conbot-has-data? cb) (conbot-has-data?-set! cb #f)
                 (conbot-search cb regx from-row start-pos)))

           ((fx>? (conbot-timeout cb) (time-millisecond diff))
            (let ([remain (fx- (conbot-timeout cb) (time-millisecond diff))])
              (conbot-read cb remain)
              (loop (time-difference (current-time) start-time))))
           (else #f)))) ]))

(define (conbot-write-char cb ch)
  (vterm-write-char (conbot-vterm cb) ch))

(define (conbot-write-key cb key)
  (vterm-write-key (conbot-vterm cb) key))

(define (conbot-write cb str)
  (vterm-write-string (conbot-vterm cb) str))

(define (conbot-clear cb)
  (let* ([vt (conbot-vterm cb)]
         [scr (vterm-screen vt)])
    (conbot-prompt-row-set! cb #f)
    (let loop ([row (conbot-first-row cb)])
      (when (not (eqv? row (conbot-top-row cb)))
        (let ([next (text-row-next row)])
          (text-row-remove row)
          (loop next))))
    (vterm-screen-erase scr (make-vterm-rect 0 0 (vterm-rows vt) (vterm-cols vt)))))

(define conbot-wait-prompt
  (case-lambda
    [(cb)
     (conbot-wait-prompt cb (conbot-prompt cb)) ]

    [(cb regex)
     (assert regex)
     (let ([prompt-row (or (conbot-prompt-row cb) (conbot-first-row cb))]
           [prompt-pos (conbot-prompt-pos cb)])
       (let ([m (conbot-wait-match cb regex prompt-row prompt-pos)])
         (when m
           (conbot-prompt-row-set! cb (conbot-cursor-row cb))
           (conbot-prompt-pos-set! cb (vterm-cursor-col (conbot-vterm cb))))
         m)) ]))

(define (split-lines str)
  (let loop ([start 0] [pos 0] [ls '()])
    (cond ((fx<? pos (string-length str))
           (let ([ch (string-ref str pos)])
             (cond ((eqv? ch #\newline)
                    (let ([s (substring str start pos)])
                      (loop (fx+ pos 1) (fx+ pos 1) (append ls (list s)))))
                   (else (loop start (fx+ pos 1) ls)))))
          (else ls))))

(define conbot-fetch-lines
  (case-lambda
    [(cb start-row)
     (conbot-fetch-lines cb start-row (conbot-cursor-row cb)) ]

    [(cb start-row end-row)
     (let loop ([curr-row start-row] [str ""])
       (cond ((not (eqv? curr-row end-row))
              (let ([row-str (text-row-string curr-row)])
                (loop (text-row-next curr-row) (string-append str row-str))))
             (else (split-lines str)))) ]))

(define conbot-fetch-lines-by-prompt
  (case-lambda
    [(cb)
     (conbot-fetch-lines-by-prompt cb (conbot-prompt cb)) ]

    [(cb prompt)
     (assert prompt)
     (let ([from-row (conbot-cursor-row cb)])
       (assert (conbot-wait-prompt cb))
       (conbot-fetch-lines cb (text-row-next from-row))) ]))

(define (conbot-do-shell cb cmd)
  (conbot-write cb cmd)
  (when (conbot-shell-eol cb)
    (conbot-write cb (conbot-shell-eol cb)))
  (conbot-fetch-lines-by-prompt cb))

)
