;; Copyright (c) 2023 
;; SPDX-License-Identifier: MIT
#!r6rs

(library (conbot private io)
  (export
    io-init
    poll-io-events
    make-io-poller
    io-poller-add-object
    io-poller-del-object
    io-poll-fd)
  (import (rnrs)
          (pffi))

(define libpty (open-shared-object "libpty.so"))

(define setup_signals (foreign-procedure libpty int setup_signals ()))
(define io_poll (foreign-procedure libpty int io_poll (int int)))
(define io_poll_bitmap (foreign-procedure libpty int io_poll (int pointer pointer)))

(define (io-init)
  (setup_signals))

(define-record-type io-poller
  (fields fd-bitmap (mutable fd-array) handler ht)
  (protocol
    (lambda (new)
      (lambda (handler)
        (new (make-bytevector (fx* 4 32) 0) #f handler
                              (make-eqv-hashtable))
      ))))

(define (update-fd-array! poller)
  (let ([bv (make-bytevector (fx* (hashtable-size (io-poller-ht poller)) 4))]
        [fds (hashtable-keys (io-poller-ht poller))])
    (do ([i 0 (fx+ i 1)]) ((fx=? i (vector-length fds)))
      (bytevector-u32-native-set! bv (fx* i 4) (vector-ref fds i)))
    (io-poller-fd-array-set! poller bv)))

(define (io-poller-add-object poller fd obj)
  (hashtable-set! (io-poller-ht poller) fd obj)
  (update-fd-array! poller))

(define (io-poller-del-object poller fd)
  (hashtable-delete! (io-poller-ht poller) fd)
  (update-fd-array! poller))

(define (io-poller-fd-ready? poller fd)
  (let ([bitmap (io-poller-fd-bitmap poller)]
        [blk (fxdiv fd 32)]
        [bit (fxmod fd 32)])
    (fxpositive? (bitwise-and (bytevector-u32-native-ref bitmap blk)
                              (fxarithmetic-shift 1 bit)))))

(define (poll-io-events poller)
  (let ([fds-num (vector-length (hashtable-keys (io-poller-ht poller)))])
    (when (fxpositive? fds-num)
      (let ([handler (io-poller-handler poller)]
            [ret (io_poll_bitmap
                         fds-num
                         (bytevector->pointer (io-poller-fd-array poller))
                         (bytevector->pointer (io-poller-fd-bitmap poller)))])
       (when (and (fxpositive? ret) handler)
         (let ([fds-num (vector-length (hashtable-keys (io-poller-ht poller)))])
           (do ([i 0 (fx+ i 1)]) ((fx=? i fds-num))
             (let ([fd (bytevector-u32-native-ref (io-poller-fd-array poller) (fx* i 4))])
               (and (io-poller-fd-ready? poller fd)
                    (handler fd (hashtable-ref (io-poller-ht poller) fd #f)))))))
       ret))))

(define (io-poll-fd fd timeout)
  (let ([ret (io_poll fd timeout)])
    (cond
      ((eqv? ret 0) 'ready)
      ((eqv? ret 1) 'timeout)
      ((eqv? ret 2) 'fail)
      ((fxnegative? ret) 'error))))

)
