/*
 * Copyright © 2004 Bruno T. C. de Oliveira
 * Copyright © 2006 Pierre Habouzit
 * Copyright © 2008-2016 Marc André Tanner
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF MIND, USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <langinfo.h>
#include <limits.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <stddef.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <termios.h>
#include <wchar.h>
#include <poll.h>
#include <time.h>
#if defined(__linux__) || defined(__CYGWIN__)
# include <pty.h>
#elif defined(__FreeBSD__) || defined(__DragonFly__)
# include <libutil.h>
#elif defined(__OpenBSD__) || defined(__NetBSD__) || defined(__APPLE__)
# include <util.h>
#endif

#ifdef _AIX
# include "forkpty-aix.c"
#elif defined __sun
# include "forkpty-sunos.c"
#endif

#define MAX(a,b) \
	   ({ __typeof__ (a) _a = (a); \
	      __typeof__ (b) _b = (b); \
	      _a > _b ? _a : _b; })

static sigset_t emptyset, blockset;

static void sigwinch_handler(int sig) {
}

static void sigterm_handler(int sig) {
}

static void sigchld_handler(int sig) {
	int status;
	pid_t pid;

	while ((pid = waitpid(-1, &status, WNOHANG)) != 0) {
		if (pid == -1) {
			if (errno == ECHILD) {
				/* no more child processes */
				break;
			}
			break;
		}
	}
}

int setup_signals(void)
{
	struct sigaction sa;
	memset(&sa, 0, sizeof sa);
	sa.sa_flags = 0;
	sigemptyset(&sa.sa_mask);
	sa.sa_handler = sigwinch_handler;
	sigaction(SIGWINCH, &sa, NULL);
	sa.sa_handler = sigchld_handler;
	sigaction(SIGCHLD, &sa, NULL);
	sa.sa_handler = sigterm_handler;
	sigaction(SIGTERM, &sa, NULL);
	sa.sa_handler = SIG_IGN;
	sigaction(SIGPIPE, &sa, NULL);

	sigemptyset(&emptyset);
	sigemptyset(&blockset);
	sigaddset(&blockset, SIGWINCH);
	sigaddset(&blockset, SIGCHLD);
	sigprocmask(SIG_BLOCK, &blockset, NULL);
}

int io_poll(int fd, int timeout)
{
	struct pollfd fds[1];
	fds[0].fd = fd;
	fds[0].events = POLLIN;

	while (true) {
		int ret = poll(fds, 1, timeout);
		if (ret < 0) {
			if (errno == EINTR)
				continue;
			return -errno;
		} else if (ret == 0) {
			return 1;
		} else if (fds[0].revents & POLLIN) {
			return 0;
		} else {
			return 2;
		}
	}
}

int io_poll_bitmap(int n, int *fds, uint32_t *bitmap)
{
	int r, nfds = 0;
	fd_set rd;

	FD_ZERO(&rd);
	memset(bitmap, 0, 32 * sizeof(uint32_t));

	for (int i = 0; i < n; i++) {
		int fd = fds[i];

		nfds = MAX(fd, nfds);
		FD_SET(fd, &rd);
	}

	if (nfds >= 32 * 32)
		return -1;

	while (true) {
		r = pselect(nfds + 1, &rd, NULL, NULL, NULL, &emptyset);
		if (r < 0) {
			if (errno == EINTR)
				continue;
			return -errno;
		}

		for (int i = 0; i < n && r > 0; i++) {
			int fd = fds[i];

			if (FD_ISSET(fd, &rd)) {
				bitmap[fd / 32] |= 1 << (fd % 32);
			}
		}
		return r;
	}
	return 0;
}

pid_t pty_open(int *pty, const char *p, int rows, int cols, const char *argv[], const char *cwd, const char *env[], int *to, int *from)
{
	int vt2ed[2], ed2vt[2];
	struct winsize ws;
	ws.ws_row = rows;
	ws.ws_col = cols;
	ws.ws_xpixel = ws.ws_ypixel = 0;

	if (to && pipe(vt2ed)) {
		*to = -1;
		to = NULL;
	}
	if (from && pipe(ed2vt)) {
		*from = -1;
		from = NULL;
	}

	pid_t pid = forkpty(pty, NULL, NULL, &ws);
	if (pid < 0)
		return -1;

	if (pid == 0) {
		setsid();

		sigset_t emptyset;
		sigemptyset(&emptyset);
		sigprocmask(SIG_SETMASK, &emptyset, NULL);

		if (to) {
			close(vt2ed[1]);
			dup2(vt2ed[0], STDIN_FILENO);
			close(vt2ed[0]);
		}

		if (from) {
			close(ed2vt[0]);
			dup2(ed2vt[1], STDOUT_FILENO);
			close(ed2vt[1]);
		}

		int maxfd = sysconf(_SC_OPEN_MAX);
		for (int fd = 3; fd < maxfd; fd++)
			if (close(fd) == -1 && errno == EBADF)
				break;

		for (const char **envp = env; envp && envp[0]; envp += 2)
			setenv(envp[0], envp[1], 1);
		//setenv("TERM", vt_term, 1);

		if (cwd) {
			int err = chdir(cwd);
			if (err) {
				fprintf(stderr, "\nchdir() failed. ");
				perror(cwd);
				exit(1);
			}
		}

		struct sigaction sa;
		memset(&sa, 0, sizeof sa);
		sa.sa_flags = 0;
		sigemptyset(&sa.sa_mask);
		sa.sa_handler = SIG_DFL;
		sigaction(SIGPIPE, &sa, NULL);

		execvp(p, (char *const *)argv);
		fprintf(stderr, "\nexecv() failed.\nCommand: '%s'\n", argv[0]);
		exit(1);
	}

	if (to) {
		close(vt2ed[0]);
		*to = vt2ed[1];
	}

	if (from) {
		close(ed2vt[1]);
		*from = ed2vt[0];
	}

	return pid;
}

void pty_resize(pid_t pid, int pty, int rows, int cols)
{
	struct winsize ws = { .ws_row = rows, .ws_col = cols };

	if (rows <= 0 || cols <= 0)
		return;

	ioctl(pty, TIOCSWINSZ, &ws);
	kill(-pid, SIGWINCH);
}

uint64_t get_time_mls(void)
{
	struct timespec ts;
	if (clock_gettime(CLOCK_REALTIME, &ts) == -1) {
		return 1;
	}

	return (uint64_t)ts.tv_sec * 1000 + (uint64_t)ts.tv_nsec / 1000000;
}

int msleep(long msec) {
    struct timespec ts;
    int res;

    if (msec < 0) {
        errno = EINVAL; // Invalid argument
        return -1;
    }

    ts.tv_sec = msec / 1000;             // Seconds part
    ts.tv_nsec = (msec % 1000) * 1000000; // Nanoseconds part

    do {
        res = nanosleep(&ts, &ts);
    } while (res && errno == EINTR);

    return res;
}

int process_wait(int pid, int timeout)
{
	uint64_t start_time = get_time_mls();
	int status;

	while (get_time_mls() - start_time < timeout) {
		pid_t result = waitpid(pid, &status, WNOHANG);
		if (result == pid) {
			if (WIFEXITED(status))
				return WEXITSTATUS(status);
			if (WIFSIGNALED(status))
				return (1 << 8) | WTERMSIG(status);
			return (2 << 8);
		} else if (result == 0) {
			msleep(100);
		} else {
			return (3 << 8);
		}
	}

	if (get_time_mls() - start_time >= timeout) {
		return (4 << 8);
	}
	return (5 << 8);
}
