#!/usr/bin/env scheme-script
;; -*- mode: scheme; coding: utf-8 -*- !#
;; Copyright (c) 2025 Vadym Kochan
;; SPDX-License-Identifier: MIT
#!r6rs

(import (rnrs (6))
	(only (srfi :13)
              string-trim string-trim-both string-index-right string-contains string-tokenize)
        (srfi :14)
	(srfi :64 testing)
	(conbot)
	(only (conbot private pty) current-directory))

(conbot-default-timeout 2000)

(define test-directory
  (let ([prog-path (list-ref (command-line) 0)])
    (substring prog-path 0 (string-index-right prog-path #\/))))

(define (script name)
  (string-append test-directory "/scripts/" name))

(test-begin "run dummy script")
(let ([cb (open-conbot-process (script "dummy.sh"))])
  (test-equal #t (conbot-running? cb))
  (conbot-wait-exit cb)
  (test-equal #f (conbot-running? cb))
  (test-assert (conbot-wait-match cb "Hello")
  (test-assert (conbot-close cb))
  )
)
(test-end)

(test-begin "run interactive script")
(let ([cb (open-conbot-process (script "interact.sh"))])
  (test-equal #t (conbot-running? cb))
  (test-assert (conbot-wait-match cb "shell>"))
  (conbot-write cb "test\n")
  (test-assert (conbot-match? (conbot-wait-match cb "hello: test")))
  (conbot-clear cb)
  (test-equal #f (conbot-wait-match cb "hello: test"))
)
(test-end)

(test-begin "run scrolling script")
(let ([cb (open-conbot-process (script "scroll.sh"))])
  (conbot-wait-exit cb)
  (conbot-update cb)

  (test-equal "Line 1" (string-trim-both (text-row-string (conbot-first-row cb))))
  (test-equal "Line 1" (string-trim-both (text-row-string (conbot-rows-ref cb 0))))
  (test-equal "Line 2" (string-trim-both (text-row-string (conbot-rows-ref cb 1))))
  (test-equal "Line 3" (string-trim-both (text-row-string (conbot-rows-ref cb 2))))
  (test-equal "Line 4" (string-trim-both (text-row-string (conbot-rows-ref cb 3))))
  (test-equal "Line 5" (string-trim-both (text-row-string (conbot-rows-ref cb 4))))
  (test-equal "Line 6" (string-trim-both (text-row-string (conbot-rows-ref cb 5))))

  (test-assert (eqv? (conbot-rows-ref cb 6) (conbot-screen-ref cb 0)))

  (test-equal "Line 7" (string-trim-both (text-row-string (conbot-screen-ref cb 0))))
  (test-equal "Line 8" (string-trim-both (text-row-string (conbot-screen-ref cb 1))))
  (test-equal "Line 9" (string-trim-both (text-row-string (conbot-screen-ref cb 2))))
  (test-equal "Line 10" (string-trim-both (text-row-string (conbot-screen-ref cb 3))))
  (test-equal "Line 11" (string-trim-both (text-row-string (conbot-screen-ref cb 4))))
  (test-equal "Line 12" (string-trim-both (text-row-string (conbot-screen-ref cb 5))))
  (test-equal "Line 13" (string-trim-both (text-row-string (conbot-screen-ref cb 6))))
  (test-equal "Line 14" (string-trim-both (text-row-string (conbot-screen-ref cb 7))))
  (test-equal "Line 15" (string-trim-both (text-row-string (conbot-screen-ref cb 8))))
  (test-equal "Line 16" (string-trim-both (text-row-string (conbot-screen-ref cb 9))))
  (test-equal "Line 17" (string-trim-both (text-row-string (conbot-screen-ref cb 10))))
  (test-equal "Line 18" (string-trim-both (text-row-string (conbot-screen-ref cb 11))))
  (test-equal "Line 19" (string-trim-both (text-row-string (conbot-screen-ref cb 12))))
  (test-equal "Line 20" (string-trim-both (text-row-string (conbot-screen-ref cb 13))))
  (test-equal "Line 21" (string-trim-both (text-row-string (conbot-screen-ref cb 14))))
  (test-equal "Line 22" (string-trim-both (text-row-string (conbot-screen-ref cb 15))))
  (test-equal "Line 23" (string-trim-both (text-row-string (conbot-screen-ref cb 16))))
  (test-equal "Line 24" (string-trim-both (text-row-string (conbot-screen-ref cb 17))))
  (test-equal "Line 25" (string-trim-both (text-row-string (conbot-screen-ref cb 18))))
  (test-equal "Line 26" (string-trim-both (text-row-string (conbot-screen-ref cb 19))))
  (test-equal "Line 27" (string-trim-both (text-row-string (conbot-screen-ref cb 20))))
  (test-equal "Line 28" (string-trim-both (text-row-string (conbot-screen-ref cb 21))))
  (test-equal "Line 29" (string-trim-both (text-row-string (conbot-screen-ref cb 22))))
  (test-equal "Line 30" (string-trim-both (text-row-string (conbot-screen-ref cb 23))))
  ;; the last is a new line
  (test-equal "" (string-trim-both (text-row-string (conbot-screen-ref cb 24))))
  (test-equal "" (string-trim-both (text-row-string (conbot-last-row cb))))
  (test-assert (eqv? (conbot-last-row cb) (conbot-screen-ref cb 24)))

  (conbot-clear cb)
  (test-assert (eqv? (conbot-rows-ref cb 0) (conbot-screen-ref cb 0)))
  (test-equal "" (string-trim-both (text-row-string (conbot-rows-ref cb 0))))
  (test-equal "" (string-trim-both (conbot-screen-string cb)))
)
(test-end)

(test-begin "run shell script")
(let ([cb (open-conbot-process (script "shell.sh"))])
  (conbot-prompt-set! cb "shell>")
  (test-assert (conbot-wait-prompt cb))
  (let ([from-row (conbot-cursor-row cb)])
    (conbot-write cb "cmd\n")
    (test-assert (conbot-wait-prompt cb))
    (let ([ls (conbot-fetch-lines cb (text-row-next from-row))])
      (test-equal 3 (length ls))
      (test-equal "Value one" (list-ref ls 0))
      (test-equal "Value two" (list-ref ls 1))
      (test-equal "Value three                                                               is kind of long" (list-ref ls 2)))
  )
  (test-equal #t (conbot-running? cb))
  (conbot-close cb)
  (test-equal #f (conbot-running? cb))
)
(test-end)

(test-begin "run shell script #2")
(let ([cb (open-conbot-process (script "shell.sh"))])
  (conbot-prompt-set! cb "shell>")
  (test-assert (conbot-wait-prompt cb))
  (conbot-write cb "cmd1\n")
  (let ([ls (conbot-fetch-lines-by-prompt cb)])
    (test-equal 3 (length ls))
    (test-equal "Value one" (list-ref ls 0))
    (test-equal "Value two" (list-ref ls 1))
    (test-equal "Value three                                                               is kind of long" (list-ref ls 2)))

  (let ([ls (conbot-do-shell cb "cmd2")])
    (test-equal 2 (length ls))
    (test-equal "Other one" (list-ref ls 0))
    (test-equal "Other two" (list-ref ls 1)))

  (let ([ls (conbot-do-shell cb "cmd3")])
    (test-assert (null? ls)))
)
(test-end)

(exit (if (zero? (test-runner-fail-count (test-runner-get))) 0 1))
