;;; R6RS ELF parser

;;;; Copyright (C)  2011, 2012, 2013, 2014, 2015 Free Software Foundation, Inc.
;;;; Copyright (C)  2025 Nikolaos Chatzikonstantnou
;;;; This program is free software: you can redistribute it and/or modify
;;;; it under the terms of the GNU General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or
;;;; (at your option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful,
;;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;;; GNU General Public License for more details.
;;;;
;;;; You should have received a copy of the GNU General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;;; Code:

(library (parser elf)
  (export

   ;; ELF e_ident record.
   <elf-ident> make-elf-ident elf-ident?
   ;; Accessors.
   elf-ident-magic elf-ident-class      elf-ident data elf-ident-version
   elf-ident-osabi elf-ident-abiversion elf-ident-pad

   ;; ELF header record.
   <elf-header> make-elf-header elf-header?
   ;; Accessors.
   elf-header-ident     elf-header-type      elf-header-machine
   elf-header-entry     elf-header-phoff     elf-header-shoff
   elf-header-flags     elf-header-ehsize    elf-header-phentsize
   elf-header-phnum     elf-header-shentsize elf-header-shnum
   elf-header-shstrndx

   ;; ELF region header record.
   <elf-region-header> make-elf-region-header elf-region-header?
   ;; Accessors.
   elf-region-header-offset elf-region-header-size

   ;; ELF program header record.
   <elf-program-header> make-elf-program-header elf-program-header?
   ;; Accessors.
   elf-program-header-type   elf-program-header-offset
   elf-program-header-vaddr  elf-program-header-paddr
   elf-program-header-filesz elf-program-header-memsz
   elf-program-header-flags  elf-program-header-align
   ;; Constructors
   make-elf32-program-header make-elf64-program-header

   ;; ELF section header record.
   <elf-section-header> make-elf-section-header elf-section-header?
   ;; Accessors.
   elf-section-header-name      elf-section-header-type
   elf-section-header-flags     elf-section-header-addr
   elf-section-header-offset    elf-section-header-size
   elf-section-header-link      elf-section-header-info
   elf-section-header-addralign elf-section-header-entsize

   ;; ELF symbol record.
   <elf-sym> make-elf-sym elf-sym?
   ;; Accessors.
   elf-sym-name elf-sym-value elf-sym-size
   elf-sym-info elf-sym-other elf-sym-shndx
   ;; Constructors.
   make-elf32-symbol make-elf64-symbol

   ;; ELF relocation record.
   <elf-rel> make-elf-rel elf-rel?
   ;; Accessors.
   elf-rel-info elf-rel-offset

   ;; ELF relocation record with addend.
   <elf-rela> make-elf-rela elf-rela?
   ;; Accessors.
   elf-rela-info elf-rela-offset elf-rela-addend

   ;; ELF dynamic record.
   <elf-dyn> make-elf-dyn elf-dyn?
   ;; Accessors.
   elf-dyn-tag elf-dyn-val

   ;; ELF note header record.
   <elf-nhdr> make-elf-nhdr elf-nhdr?
   ;; Accessors.
   elf-nhdr-namesz elf-nhdr-descsz elf-nhdr-type

   ;; ELF note record.
   <elf-note> make-elf-note elf-note?
   ;; Accessors
   elf-note-name elf-note-desc

   ;; Predicates
   valid-elf-ident?
   valid-elf-header?

   ;; Reading records.
   read-elf-ident
   read-elf-header
   read-elf-program-header read-elf-segment for-each-program-header
   read-elf-section-header read-elf-section for-each-section-header
                           read-elf-sym     for-each-sym
                           read-elf-rel     for-each-rel
                           read-elf-rela    for-each-rela
                           read-elf-dyn     for-each-dyn
   read-elf-nhdr           read-elf-note    for-each-note

   ;; Seeking to a program header or section header.
   seek-to-program-header!
   seek-to-section-header!

   ;; Reading bytes into a bytevector.
   read-elf-ident-bytes!
   read-elf-header-bytes!
   read-elf-program-header-bytes!
   read-elf-segment-bytes!
   read-elf-section-header-bytes!
   read-elf-section-bytes!
   read-elf-sym-bytes!
   read-elf-rel-bytes!
   read-elf-rela-bytes!
   read-elf-dyn-bytes!
   read-elf-nhdr-bytes!
   read-elf-note-bytes!

   ;; Parsing
   parse-elf-ident          parse-elf-ident-len
   parse-elf-header         parse-elf-header-len
   parse-elf-program-header parse-elf-program-header-len
   parse-elf-section-header parse-elf-section-header-len
   parse-elf-sym            parse-elf-sym-len
   parse-elf-rel            parse-elf-rel-len
   parse-elf-rela           parse-elf-rela-len
   parse-elf-dyn            parse-elf-dyn-len
   parse-elf-nhdr           parse-elf-nhdr-len
   parse-elf-note)

  (import (rnrs base (6))
          (rnrs arithmetic bitwise (6))
          (rnrs bytevectors (6))
          (rnrs control (6))
          (rnrs io ports (6))
          (rnrs records syntactic (6))
          (rnrs records procedural (6))
          (rnrs records inspection (6))
          (rnrs syntax-case (6))
          (parser elf cstrings)
          (parser elf enums))


;;; Records.

  (define-record-type (<elf-ident> make-elf-ident elf-ident?)
    (fields
     (immutable magic      elf-ident-magic)
     (immutable class      elf-ident-class)
     (immutable data       elf-ident-data)
     (immutable version    elf-ident-version)
     (immutable osabi      elf-ident-osabi)
     (immutable abiversion elf-ident-abiversion)
     (immutable pad        elf-ident-pad)))

  (define-record-type (<elf-header> make-elf-header elf-header?)
    (parent <elf-ident>)
    (fields
     (immutable type      elf-header-type)
     (immutable machine   elf-header-machine)
     (immutable version   elf-header-version)
     (immutable entry     elf-header-entry)
     (immutable phoff     elf-header-phoff)
     (immutable shoff     elf-header-shoff)
     (immutable flags     elf-header-flags)
     (immutable ehsize    elf-header-ehsize)
     (immutable phentsize elf-header-phentsize)
     (immutable phnum     elf-header-phnum)
     (immutable shentsize elf-header-shentsize)
     (immutable shnum     elf-header-shnum)
     (immutable shstrndx  elf-header-shstrndx))
    ;; In the protocol below we take an <elf-ident> and incorporate
    ;; its fields into the <elf-header> record.  This allows us to use
    ;; the <elf-ident> accessors in <elf-header>.
    (protocol
     (lambda (p)
       (lambda (elf-ident type machine version entry phoff shoff flags
                   ehsize phentsize phnum shentsize shnum shstrndx)
         ((apply p (record->list elf-ident))
          type machine version entry phoff shoff flags
          ehsize phentsize phnum shentsize shnum shstrndx)))))

  ;; This record is only used as a means to provide common offset and
  ;; size methods to the program header and section header records.
  (define-record-type (<elf-region-header> make-elf-region-header elf-region-header?)
    (fields
     (immutable offset elf-region-header-offset)
     (immutable size   elf-region-header-size)))

  (define-record-type (<elf-program-header> make-elf-program-header elf-program-header?)
    (parent <elf-region-header>)
    ;; The fields are in the 32-bit program header order.
    (fields
     (immutable type   elf-program-header-type)
     (immutable offset elf-program-header-offset)
     (immutable vaddr  elf-program-header-vaddr)
     (immutable paddr  elf-program-header-paddr)
     (immutable filesz elf-program-header-filesz)
     (immutable memsz  elf-program-header-memsz)
     (immutable flags  elf-program-header-flags)
     (immutable align  elf-program-header-align))
    ;; In the protocol below, we initialize the parent record with the
    ;; notion of offset and size from a program header.
    (protocol
     (lambda (p)
       (lambda (type offset vaddr paddr filesz memsz flags align)
         ((p offset filesz)
          type offset vaddr paddr filesz memsz flags align)))))

  (define-record-type (<elf-section-header> make-elf-section-header elf-section-header?)
    (parent <elf-region-header>)
    (fields
     (immutable name      elf-section-header-name)
     (immutable type      elf-section-header-type)
     (immutable flags     elf-section-header-flags)
     (immutable addr      elf-section-header-addr)
     (immutable offset    elf-section-header-offset)
     (immutable size      elf-section-header-size)
     (immutable link      elf-section-header-link)
     (immutable info      elf-section-header-info)
     (immutable addralign elf-section-header-addralign)
     (immutable entsize   elf-section-header-entsize))
    ;; In the protocol below, we initialize the parent record with the
    ;; notion of offset and size from a section header.
    (protocol
     (lambda (p)
       (lambda (name type flags addr offset size link info addralign entsize)
         ((p offset size)
          name type flags addr offset size link info addralign entsize)))))

  (define-record-type (<elf-sym> make-elf-sym elf-sym?)
    ;; The fields are in the 32-bit symbol order.
    (fields
     (immutable name  elf-sym-name)
     (immutable value elf-sym-value)
     (immutable size  elf-sym-size)
     (immutable info  elf-sym-info)
     (immutable other elf-sym-other)
     (immutable shndx elf-sym-shndx)))

  (define-record-type (<elf-rel> make-elf-rel elf-rel?)
    (fields
     (immutable offset elf-rel-offset)
     (immutable info   elf-rel-info)))

  (define-record-type (<elf-rela> make-elf-rela elf-rela?)
    (fields
     (immutable offset elf-rela-offset)
     (immutable info   elf-rela-info)
     (immutable addend elf-rela-addend)))

  (define-record-type (<elf-dyn> make-elf-dyn elf-dyn?)
    (fields
     (immutable tag elf-dyn-tag)
     (immutable val elf-dyn-val)))

  (define-record-type (<elf-nhdr> make-elf-nhdr elf-nhdr?)
    (fields
     (immutable namesz elf-nhdr-namesz)
     (immutable descsz elf-nhdr-descsz)
     (immutable type   elf-nhdr-type)))

  (define-record-type (<elf-note> make-elf-note elf-note?)
    (parent <elf-nhdr>)
    (fields
     (immutable name elf-note-name)
     (immutable desc elf-note-desc))
    ;; In the protocol below we take an <elf-nhdr> and incorporate its
    ;; fields into the <elf-note> record.  This allows us to use the
    ;; <elf-nhdr> accessors in <elf-note>.
    (protocol
     (lambda (p)
       (lambda (elf-nhdr name desc)
         ((apply p (record->list elf-nhdr))
          name desc)))))


;;; Macros.

  ;; Just like get-bytevector-n! but throws an error if the requested
  ;; number of bytes was not read.
  (define-syntax my-get-bytevector-n!
    (syntax-rules ()
      ((my-get-bytevector-n! port bv start size error-message)
       (let ((n (get-bytevector-n! port bv start size)))
         (when (or (eof-object? n)
                   (not (= n size)))
           (error error-message n))))))

  ;; This macro and byte-spec-len are useful for define-elf-parser
  ;; below.
  (define-syntax byte-spec
    (syntax-rules (u8 u16 u32 u64
                   s8 s16 s32 s64)
      ((byte-spec bv i endianness) '())
      ;; Unsigned
      ((byte-spec bv i endianness u8 rest ...)
       (cons (bytevector-u8-ref bv i)
             (byte-spec bv (+ i 1) endianness rest ...)))
      ((byte-spec bv i endianness u16 rest ...)
       (cons (bytevector-u16-ref bv i endianness)
             (byte-spec bv (+ i 2) endianness rest ...)))
      ((byte-spec bv i endianness u32 rest ...)
       (cons (bytevector-u32-ref bv i endianness)
             (byte-spec bv (+ i 4) endianness rest ...)))
      ((byte-spec bv i endianness u64 rest ...)
       (cons (bytevector-u64-ref bv i endianness)
             (byte-spec bv (+ i 8) endianness rest ...)))
      ;; Signed
      ((byte-spec bv i endianness s8 rest ...)
       (cons (bytevector-s8-ref bv i)
             (byte-spec bv (+ i 1) endianness rest ...)))
      ((byte-spec bv i endianness s16 rest ...)
       (cons (bytevector-s16-ref bv i endianness)
             (byte-spec bv (+ i 2) endianness rest ...)))
      ((byte-spec bv i endianness s32 rest ...)
       (cons (bytevector-s32-ref bv i endianness)
             (byte-spec bv (+ i 4) endianness rest ...)))
      ((byte-spec bv i endianness s64 rest ...)
       (cons (bytevector-s64-ref bv i endianness)
             (byte-spec bv (+ i 8) endianness rest ...)))))

  (define-syntax byte-spec-len
    (syntax-rules (u8 u16 u32 u64
                   s8 s16 s32 s64)
      ((byte-spec-len) 0)
      ((byte-spec-len u8  rest ...) (+ 1 (byte-spec-len rest ...)))
      ((byte-spec-len s8  rest ...) (+ 1 (byte-spec-len rest ...)))
      ((byte-spec-len u16 rest ...) (+ 2 (byte-spec-len rest ...)))
      ((byte-spec-len s16 rest ...) (+ 2 (byte-spec-len rest ...)))
      ((byte-spec-len u32 rest ...) (+ 4 (byte-spec-len rest ...)))
      ((byte-spec-len s32 rest ...) (+ 4 (byte-spec-len rest ...)))
      ((byte-spec-len u64 rest ...) (+ 8 (byte-spec-len rest ...)))
      ((byte-spec-len s64 rest ...) (+ 8 (byte-spec-len rest ...)))))

  ;; Defines a parser given a specification. In particular
  ;; (define-elf-parser X ...) will define two procedures, parse-X
  ;; which parses an object instance of the <X> record out of a
  ;; bytevector, and parse-X-len, which returns the number of bytes in
  ;; size that must be available in the bytevector given the class
  ;; (ELFCLASS32 or ELFCLASS64).
  (define-syntax define-elf-parser
    (lambda (stx)
      (define (symbol-append . rest)
        (string->symbol
         (apply string-append
                (map symbol->string rest))))
      (syntax-case stx ()
        ((define-elf-parser x y z) #'(define-elf-parser x "" y z))
        ((define-elf-parser name docstring
           (constructor32 rest32 ...)
           (constructor64 rest64 ...))
         (with-syntax ((parse-name
                        (datum->syntax #'name
                                       (symbol-append 'parse- (syntax->datum #'name))))
                       (parse-name-len
                        (datum->syntax #'name
                                       (symbol-append 'parse- (syntax->datum #'name) '-len))))
           #'(begin
               (define (parse-name elf-header bv start)
                 docstring
                 (let ((class (elf-ident-class elf-header))
                       (endianness (data->endianness (elf-ident-data elf-header))))
                   (if (= class ELFCLASS32)
                       (apply constructor32
                              (byte-spec bv start endianness
                                         rest32 ...))
                       (apply constructor64
                              (byte-spec bv start endianness
                                         rest64 ...)))))
               (define (parse-name-len class)
                 (if (= class ELFCLASS32)
                     (byte-spec-len rest32 ...)
                     (byte-spec-len rest64 ...)))))))))


;;; Constructors.
;;;
;;; These constructors are useful because the 32-bit ELF structure and
;;; the 64-bit ELF structure have their fields in different order in
;;; the ELF specification.  Because we only provide a single Scheme
;;; record, these constructors reorder the arguments to match the
;;; given records' constructor.

  (define (make-elf32-program-header type offset vaddr paddr filesz memsz flags align)
    (make-elf-program-header         type offset vaddr paddr filesz memsz flags align))

  (define (make-elf64-program-header type flags offset vaddr paddr filesz memsz align)
    (make-elf-program-header type offset vaddr paddr filesz memsz flags align))

  (define (make-elf32-symbol name value size info other shndx)
    (make-elf-sym         name value size info other shndx))

  (define (make-elf64-symbol name info other shndx value size)
    (make-elf-sym         name value size info other shndx))


;;; Seeking to headers.
;;;
;;; We do not validate the INDEX argument to be within bounds, instead
;;; we trust the user.  This makes it easier to access indices larger
;;; than PN_XNUM (for program headers) or SHN_LORESERVE (for section
;;; headers).

  (define (seek-to-program-header! elf-header port index)
    "Seek PORT to the INDEX entry in the program header table."
    (let ((phoff     (elf-header-phoff     elf-header))
          (phentsize (elf-header-phentsize elf-header)))
      (set-port-position! port (+ phoff (* index phentsize)))))

  (define (seek-to-section-header! elf-header port index)
    "Seek PORT to the INDEX entry in the section header table."
    (let ((shoff     (elf-header-shoff     elf-header))
          (shentsize (elf-header-shentsize elf-header)))
      (set-port-position! port (+ shoff (* index shentsize)))))


;;; ELF e_ident.

  (define parse-elf-ident-len EI_NIDENT)

  (define (read-elf-ident-bytes! port bv start)
    "Read an ELF e_ident from PORT into BV.

There must be room for at least parse-elf-ident-len bytes"
    (my-get-bytevector-n! port bv start EI_NIDENT
                          "Cannot read e_ident"))

  (define (read-elf-ident port)
    "Read an <elf-ident> from PORT."
    (let ((bv (make-bytevector parse-elf-ident-len)))
      (read-elf-ident-bytes! port bv 0)
      (parse-elf-ident bv 0)))


;;; ELF headers.

  (define elf32-header-len (+ EI_NIDENT 36))
  (define elf64-header-len (+ EI_NIDENT 48))
  (define (parse-elf-header-len class)
    (if (= ELFCLASS32 class)
        elf32-header-len
        elf64-header-len))

  (define (read-elf-header-bytes! port bv start)
    "Read an ELF header (32 or 64-bit) from PORT into BV.

There must be room for at least elf64-header-len bytes."
    (read-elf-ident-bytes! port bv start)
    (let* ((class (bytevector-u8-ref bv (+ start EI_CLASS)))
           (start (+ start EI_NIDENT))
           (rest (- (parse-elf-header-len class) EI_NIDENT)))
      (my-get-bytevector-n! port bv start rest
                            "Cannot read ELF header.")))

  (define (read-elf-header port)
    "Read an <elf-header> from PORT."
    (let ((bv (make-bytevector elf64-header-len)))
      (read-elf-header-bytes! port bv 0)
      (parse-elf-header bv 0)))


;;; ELF program headers.

  (define (read-elf-program-header-bytes! elf-header port bv start)
    "Read an ELF program header (32 or 64-bit) from PORT into BV."
    (let ((phentsize (elf-header-phentsize elf-header)))
      (my-get-bytevector-n! port bv start phentsize
                         "Cannot read ELF program header.")))

  (define (read-elf-program-header elf-header port)
    "Read an elf-program-header from PORT."
    (let* ((phentsize (elf-header-phentsize elf-header))
           (bv (make-bytevector phentsize)))
      (read-elf-program-header-bytes! elf-header port bv 0)
      (parse-elf-program-header elf-header bv 0)))

  (define (for-each-program-header elf-header port f)
    "Invoke F on each program header."
    (let ((phentsize (elf-header-phentsize elf-header))
          (phnum     (elf-header-phnum     elf-header))
          (phoff     (elf-header-phoff     elf-header)))
      (unless (zero? phoff)
        (when (= PN_XNUM phnum)
          ;; Real phnum is in first section header sh_info.
          (set-port-position! port (elf-header-shoff elf-header))
          (let ((first-section-header (read-elf-section-header elf-header port)))
            (set! phnum (elf-section-header-info first-section-header))))
        (unless (zero? phnum)
          (set-port-position! port phoff)
          (let ((bv (make-bytevector phentsize)))
            (do ((i 0 (+ i 1)))
                ((>= i phnum))
              (read-elf-program-header-bytes! elf-header port bv 0)
              (let ((ph (parse-elf-program-header elf-header bv 0)))
                (f ph))))))
      (values)))


;;; ELF segments.

  (define (read-elf-segment-bytes! program-header port bv start)
    "Read an ELF segment from PORT into BV."
    (let ((filesz (elf-program-header-filesz program-header)))
      (my-get-bytevector-n! port bv start filesz
                         "Cannot read ELF segment.")))

  (define (read-elf-segment program-header port)
    "Read an ELF segment from PORT."
    (let* ((filesz (elf-program-header-filesz program-header))
           (offset (elf-program-header-offset program-header))
           (bv (make-bytevector filesz)))
      (set-port-position! port offset)
      (read-elf-segment-bytes! program-header port bv 0)
      bv))


;;; ELF section headers.

  (define (read-elf-section-header-bytes! elf-header port bv start)
    "Read an ELF section header (32 or 64-bit) from PORT into BV."
    (let ((shentsize (elf-header-shentsize elf-header)))
      (my-get-bytevector-n! port bv start shentsize
                         "Cannot read ELF section header.")))

  (define (read-elf-section-header elf-header port)
    "Read an elf-section-header from PORT."
    (let* ((shentsize (elf-header-shentsize elf-header))
           (bv (make-bytevector shentsize)))
      (read-elf-section-header-bytes! elf-header port bv 0)
      (parse-elf-section-header elf-header bv 0)))

  (define (for-each-section-header elf-header port f)
    "Invoke F on each section header."
    (let ((shentsize (elf-header-shentsize elf-header))
          (shnum     (elf-header-shnum     elf-header))
          (shoff     (elf-header-shoff     elf-header)))
      (unless (zero? shoff)
        (set-port-position! port shoff)
        (let ((bv (make-bytevector shentsize)))
          ;; The first section is special.
          (read-elf-section-header-bytes! elf-header port bv 0)
          (let ((first-section-header (parse-elf-section-header elf-header bv 0)))
            (when (zero? shnum)
              ;; Real shnum is in first section header sh_size.
              (set! shnum (elf-section-header-size first-section-header)))
            ;; Invoke F on the first section.
            (f first-section-header))
          ;; We start at 1 because we have already read the first
          ;; entry of the section header table.
          (do ((i 1 (+ i 1)))
              ((>= i shnum))
            (read-elf-section-header-bytes! elf-header port bv 0)
            (let ((sh (parse-elf-section-header elf-header bv 0)))
              (f sh)))))
      (values)))


;;; ELF sections.

  (define (read-elf-section-bytes! section-header port bv start)
    "Read an ELF section from PORT into BV."
    (let ((size (elf-section-header-size section-header)))
      (my-get-bytevector-n! port bv start size
                            "Cannot read ELF section.")))

  (define (read-elf-section section-header port)
    "Read an ELF section from PORT."
    (let ((size   (elf-section-header-size   section-header))
          (offset (elf-section-header-offset section-header))
          (type   (elf-section-header-type   section-header)))
      (if (or (= SHT_NOBITS type)
              (zero? size))
          #vu8()
          (let ((bv (make-bytevector size)))
            (set-port-position! port offset)
            (read-elf-section-bytes! section-header port bv 0)
            bv))))


;;; ELF symbols.

  (define (read-elf-sym-bytes! elf-header port bv start)
    "Read an ELF symbol from PORT into BV."
    (let* ((class (elf-ident-class elf-header))
           (sym-len (parse-elf-sym-len class)))
      (my-get-bytevector-n! port bv start sym-len
                            "Cannot read ELF symbol.")))

  (define (read-elf-sym elf-header port)
    "Read an ELF symbol from PORT."
    (let* ((class (elf-ident-class elf-header))
           (sym-len (parse-elf-sym-len class))
           (bv (make-bytevector sym-len)))
      (read-elf-sym-bytes! elf-header port bv 0)
      (parse-elf-sym elf-header bv 0)))

  (define (for-each-sym elf-header section-header port f)
    "Invoke F on each ELF symbol from SECTION-HEADER."
    (let* ((class (elf-ident-class elf-header))
           (sym-len (parse-elf-sym-len class))
           (offset (elf-section-header-offset section-header))
           (size   (elf-section-header-size   section-header))
           (type   (elf-section-header-type   section-header))
           (bv (make-bytevector sym-len)))
      (set-port-position! port offset)
      (let loop ((i 0))
        (unless (>= i size)
          (read-elf-sym-bytes! elf-header port bv 0)
          (let ((sym (parse-elf-sym elf-header bv 0)))
            (f sym)
            (loop (+ i sym-len)))))))


;;; ELF relocation entries.

  (define (read-elf-rel-bytes! elf-header port bv start)
    "Read an ELF relocation entry from PORT into BV."
    (let* ((class (elf-ident-class elf-header))
           (rel-len (parse-elf-rel-len class)))
      (my-get-bytevector-n! port bv start rel-len
                            "Cannot read ELF relocation entry.")))

  (define (read-elf-rel elf-header port)
    "Read an ELF relocation entry from PORT."
    (let* ((class (elf-ident-class elf-header))
           (rel-len (parse-elf-rel-len class))
           (bv (make-bytevector rel-len)))
      (read-elf-rel-bytes! elf-header port bv 0)))

  (define (for-each-rel elf-header section-header port f)
    "Invoke F on each ELF relocation entry from SECTION-HEADER."
    (let* ((class (elf-ident-class elf-header))
           (rel-len (parse-elf-rel-len class))
           (offset (elf-section-header-offset section-header))
           (size   (elf-section-header-size   section-header))
           (type   (elf-section-header-type   section-header))
           (bv (make-bytevector rel-len)))
      (set-port-position! port offset)
      (let loop ((i 0))
        (unless (>= i size)
          (read-elf-rel-bytes! elf-header port bv 0)
          (let ((rel (parse-elf-rel elf-header bv 0)))
            (f rel)
            (loop (+ i rel-len)))))
      (values)))


;;; ELF relocation entries with addend.

  (define (read-elf-rela-bytes! elf-header port bv start)
    "Read an ELF relocation entry (with addend) from PORT into BV."
    (let* ((class (elf-ident-class elf-header))
           (rela-len (parse-elf-rela-len class)))
      (my-get-bytevector-n! port bv start rela-len
                            "Cannot read ELF relocation entry (with addend).")))

  (define (read-elf-rela elf-header port)
    "Read an ELF relocation entry (with addend) from PORT."
    (let* ((class (elf-ident-class elf-header))
           (rela-len (parse-elf-rela-len class))
           (bv (make-bytevector rela-len)))
      (read-elf-rela-bytes! elf-header port bv 0)))

  (define (for-each-rela elf-header section-header port f)
    "Invoke F on each ELF relocation entry (with addend) from SECTION-HEADER."
    (let* ((class (elf-ident-class elf-header))
           (rela-len (parse-elf-rela-len class))
           (offset (elf-section-header-offset section-header))
           (size   (elf-section-header-size   section-header))
           (type   (elf-section-header-type   section-header))
           (bv (make-bytevector rela-len)))
      (set-port-position! port offset)
      (let loop ((i 0))
        (unless (>= i size)
          (read-elf-rela-bytes! elf-header port bv 0)
          (let ((rela (parse-elf-rela elf-header bv 0)))
            (f rela)
            (loop (+ i rela-len)))))
      (values)))


;;; ELF dynamic entries.

  (define (read-elf-dyn-bytes! elf-header port bv start)
    "Read an ELF dynamic entry from PORT into BV."
    (let* ((class (elf-ident-class elf-header))
           (size (parse-elf-dyn-len class)))
      (my-get-bytevector-n! port bv 0 size
                            "Cannot read ELF dynamic entry.")))

  (define (read-elf-dyn elf-header port)
    "Read an ELF dynamic entry from PORT."
    (let* ((class (elf-ident-class elf-header))
           (size (parse-elf-dyn-len class))
           (bv (make-bytevector size)))
      (read-elf-dyn-bytes! elf-header port bv 0)
      (parse-elf-dyn elf-header bv 0)))

  (define (for-each-dyn elf-header region-header port f)
    "Invoke F on each ELF dynamic entry from REGION-HEADER.

REGION-HEADER is either a program header or a section header."
    (let* ((class (elf-ident-class elf-header))
           (dyn-len (parse-elf-dyn-len class))
           (offset (elf-region-header-offset region-header))
           (n      (elf-region-header-size   region-header))
           (bv (make-bytevector dyn-len)))
      (set-port-position! port offset)
      (let loop ((i 0))
        (unless (>= i n)
          (read-elf-dyn-bytes! elf-header port bv 0)
          (let ((dynamic-entry (parse-elf-dyn elf-header bv 0)))
            (f dynamic-entry)
            (unless (= DT_NULL (elf-dyn-tag dynamic-entry))
              (loop (+ i dyn-len))))))
      (values)))


;;; ELF note headers.

  (define (read-elf-nhdr-bytes! elf-header port bv start)
    "Read an ELF note header from PORT into BV."
    (let* ((class (elf-ident-class elf-header))
           (size (parse-elf-nhdr-len class)))
      (my-get-bytevector-n! port bv 0 size
                            "Cannot read ELF note header.")))

  (define (read-elf-nhdr elf-header port)
    "Read an ELF note header from PORT."
    (let* ((class (elf-ident-class elf-header))
           (size (parse-elf-nhdr-len class))
           (bv (make-bytevector size)))
      (read-elf-nhdr-bytes! elf-header port bv 0)
      (parse-elf-nhdr elf-header bv 0)))


;;; ELF notes.

  (define (read-elf-note-bytes! elf-header port bv start)
    "Read an ELF note (header, name and desc) from PORT into BV."
    (read-elf-nhdr-bytes! elf-header port bv start)
    (let* ((class (elf-ident-class elf-header))
           (nhdr-size (parse-elf-nhdr-len class))
           (nhdr (parse-elf-nhdr elf-header bv start))
           (namesz (elf-nhdr-namesz nhdr))
           (descsz (elf-nhdr-descsz nhdr))
           (note-size (+ (align-up-4 namesz)
                         (align-up-4 descsz))))
      (my-get-bytevector-n! port bv (+ start nhdr-size)
                            note-size
                            "Cannot read ELF note name and desc.")))

  (define (read-elf-note elf-header port)
    "Read an ELF note (header, name and desc) from PORT."
    ;; We don't use a combination of read-elf-note-bytes! and
    ;; parse-elf-note because reading the entire bytevector and then
    ;; copying its parts for name and desc is less efficient than the
    ;; implementation of read-elf-note below.
    (let* ((nhdr (read-elf-nhdr elf-header port))
           (namesz (elf-nhdr-namesz nhdr))
           (descsz (elf-nhdr-descsz nhdr))
           (name (make-bytevector namesz))
           (desc (make-bytevector descsz)))
      (unless (zero? namesz)
        (my-get-bytevector-n! port name 0 namesz
                              "Cannot read ELF note name."))
      ;; Skip 4-byte alignment padding.
      (skip-n port (- (align-up-4 namesz) namesz))
      (unless (zero? descsz)
        (my-get-bytevector-n! port desc 0 descsz
                              "Cannot read ELF note desc."))
      ;; Skip 4-byte alignment padding.
      (skip-n port (- (align-up-4 namesz) descsz))
      (let ((name-str (if (<= namesz 1) ""
                          ;; Skip the final NUL byte.
                          (parse-ascii-string name 0 (- namesz 1)))))
        (make-elf-note nhdr name-str desc))))

  (define (for-each-note elf-header region-header port f)
    "Invoke F on each ELF note from REGION-HEADER.

REGION-HEADER is either a program header or a section header."
    (let* ((class (elf-ident-class elf-header))
           (nhdr-len (parse-elf-nhdr-len class))
           (offset (elf-region-header-offset region-header))
           (n      (elf-region-header-size   region-header))
           (bv (make-bytevector nhdr-len)))
      (set-port-position! port offset)
      (let loop ((i 0))
        (unless (>= i n)
          (read-elf-nhdr-bytes! elf-header port bv 0)
          (let* ((nhdr (parse-elf-nhdr elf-header bv 0))
                 (namesz (elf-nhdr-namesz nhdr))
                 (descsz (elf-nhdr-descsz nhdr))
                 (name (make-bytevector namesz))
                 (desc (make-bytevector descsz)))
            (unless (zero? namesz)
              (my-get-bytevector-n! port name 0 namesz
                                    "Cannot read ELF note name."))
            ;; Skip 4-byte alignment padding.
            (set-port-position! port
                                (align-up-4 (port-position port)))
            (unless (zero? descsz)
              (my-get-bytevector-n! port desc 0 descsz
                                    "Cannot read ELF note desc."))
            ;; Skip 4-byte alignment padding.
            (set-port-position! port
                                (align-up-4 (port-position port)))
            (let* ((name-str (if (<= namesz 1) ""
                                 ;; Skip the final NUL byte.
                                 (parse-ascii-string name 0 (- namesz 1))))
                   (note (make-elf-note nhdr name-str desc)))
              (f note))
            (loop (+ i nhdr-len
                     (align-up-4 namesz)
                     (align-up-4 descsz))))))
      (values)))


;;; Predicates.

  (define (valid-elf-ident? elf-ident)
    "Does ELF-IDENT represent a valid ELF e_ident?"
    (and
     (elf-magic?      (elf-ident-magic      elf-ident))
     (elf-class?      (elf-ident-class      elf-ident))
     (elf-data?       (elf-ident-data       elf-ident))
     (elf-version?    (elf-ident-version    elf-ident))
     (elf-osabi?      (elf-ident-osabi      elf-ident))
     (elf-abiversion? (elf-ident-abiversion elf-ident))
     (elf-pad?        (elf-ident-pad        elf-ident))))

  (define (valid-elf-header? elf-header)
    "Does ELF-HEADER represent a valid ELF header?"
    (let ((class (elf-ident-class elf-header)))
      (and
       (valid-elf-ident? elf-header)
       (elf-type? (elf-header-type elf-header))
       (elf-machine? (elf-header-machine elf-header))
       (elf-version? (elf-header-version elf-header))
       (= (parse-elf-header-len class)
          (elf-header-ehsize elf-header)))))


;;; Parsers.

  (define (parse-elf-ident bv start)
    "Parse an ELF e_ident."
    (let ((u8 (lambda (i) (bytevector-u8-ref bv (+ i start))))
          (bv-copy (lambda (s e)
                     (let ((x (make-bytevector (- e s))))
                       (bytevector-copy! bv (+ s start) x 0 (- e s))
                       x))))
      (make-elf-ident
       (bv-copy 0 4)
       (u8 EI_CLASS)
       (u8 EI_DATA)
       (u8 EI_VERSION)
       (u8 EI_OSABI)
       (u8 EI_ABIVERSION)
       (bv-copy EI_PAD EI_NIDENT))))

  (define (parse-elf-header bv start)
    "Parse an ELF header (32 or 64-bit)."
    (let* ((elf-ident (parse-elf-ident bv start))
           (start (+ start EI_NIDENT))
           (class (elf-ident-class elf-ident))
           (endianness (data->endianness (elf-ident-data elf-ident))))
      (apply make-elf-header elf-ident
             (if (= class ELFCLASS32)
                 (byte-spec bv start endianness
                            u16 u16 u32 u32 u32
                            u32 u32 u16 u16 u16
                            u16 u16 u16)
                 (byte-spec bv start endianness
                            u16 u16 u32 u64 u64
                            u64 u32 u16 u16 u16
                            u16 u16 u16)))))

  (define-elf-parser elf-program-header
    "Parse an ELF program header (32 or 64-bit)."
    (make-elf32-program-header u32 u32 u32 u32
                               u32 u32 u32 u32)
    (make-elf64-program-header u32 u32 u64 u64
                               u64 u64 u64 u64))

  (define-elf-parser elf-section-header
    "Parse an ELF section header (32 or 64-bit)."
    (make-elf-section-header u32 u32 u32 u32 u32
                             u32 u32 u32 u32 u32)
    (make-elf-section-header u32 u32 u64 u64 u64
                             u64 u32 u32 u64 u64))

  (define-elf-parser elf-sym
    "Parse a symbol table entry."
    (make-elf32-symbol u32 u32 u32 u8 u8 u16)
    (make-elf64-symbol u32 u8 u8 u16 u64 u64))

  (define-elf-parser elf-rel
    "Parse an ELF relocation entry."
    (make-elf-rel u32 u32)
    (make-elf-rel u64 u64))

  (define-elf-parser elf-rela
    "Parse an ELF relocation entry with addend."
    (make-elf-rel u32 u32 u32)
    (make-elf-rel u64 u64 u64))

  (define-elf-parser elf-dyn
    "Parse an ELF dynamic entry."
    (make-elf-dyn s32 u32)
    (make-elf-dyn s64 u64))

  (define-elf-parser elf-nhdr
    "Parse an ELF note header."
    (make-elf-nhdr u32 u32 u32)
    (make-elf-nhdr u32 u32 u32))

  (define (parse-elf-note elf-header bv start)
    "Parse an ELF note (header, name and desc)."
    (let* ((class (elf-ident-class elf-header))
           (nhdr-size (parse-elf-nhdr-len class))
           (nhdr (parse-elf-nhdr elf-header bv 0))
           (note-start (+ start nhdr-size))
           (namesz (elf-nhdr-namesz nhdr))
           (descsz (elf-nhdr-descsz nhdr))
           (name-str (if (<= namesz 1) ""
                         (parse-ascii-string bv note-start
                                             ;; Skip the NUL byte.
                                             (+ note-start namesz -1))))
           (desc (make-bytevector descsz)))
      (bytevector-copy! bv (+ note-start (align-up-4 namesz))
                        desc 0 descsz)
      (make-elf-note nhdr name-str desc)))


;;; Auxiliary.

  (define (record->list record-object)
    "Convert the RECORD-OBJECT into a list of values of its fields."
    (let* ((rtd (record-rtd record-object))
           (names (record-type-field-names rtd))
           (n (vector-length names)))
      (let loop ((i 0))
        (if (>= i n) '()
            (cons ((record-accessor rtd i) record-object)
                  (loop (+ i 1)))))))

  (define (data->endianness data)
    "Convert ELF endianness to Scheme endianness."
    (if (= data ELFDATA2LSB)
        (endianness little)
        (endianness big)))

  (define (align-up-4 x)
    "If unaligned, align the integer X up to a multiple of 4."
    (bitwise-and (+ 3 x) (bitwise-not 3)))

  (define (skip-n port n)
    "Skip N bytes from PORT."
    (unless (<= n 0)
      (get-u8 port)
      (skip-n port (- n 1)))))
