;;;; ELF enumerations for R6RS Scheme
;;;; Copyright (C)  2025 Nikolaos Chatzikonstantnou
;;;;
;;;; This program is free software: you can redistribute it and/or modify
;;;; it under the terms of the GNU General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or
;;;; (at your option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful,
;;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;;; GNU General Public License for more details.
;;;;
;;;; You should have received a copy of the GNU General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;;; Code:

;;; TODO make sure all symbols are exported.

(library (parser elf enums)
  (export
   elf-identifier?
   EI_MAG0 EI_MAG1 EI_MAG2 EI_MAG3 EI_CLASS EI_DATA EI_VERSION EI_OSABI
   EI_ABIVERSION EI_PAD EI_NIDENT

   elf-magic?
   ELFMAG0 ELFMAG1 ELFMAG2 ELFMAG3

   elf-class?
   ELFCLASSNONE ELFCLASS32 ELFCLASS64

   elf-data?
   ELFDATANONE ELFDATA2LSB ELFDATA2MSB

   elf-version?
   EV_NONE EV_CURRENT

   elf-osabi?
   ELFOSABI_NONE ELFOSABI_SYSV ELFOSABI_HPUX ELFOSABI_NETBSD
   ELFOSABI_GNU ELFOSABI_SOLARIS ELFOSABI_AIX ELFOSABI_IRIX
   ELFOSABI_FREEBSD ELFOSABI_TRU64 ELFOSABI_MODESTO ELFOSABI_OPENBSD
   ELFOSABI_ARM ELFOSABI_ARM ELFOSABI_STANDALONE

   elf-abiversion?

   elf-pad?

   elf-type?
   ET_NONE ET_REL ET_EXEC ET_DYN ET_CORE

   elf-machine?
   EM_NONE EM_M32 EM_SPARC EM_386 EM_68K EM_88K EM_IAMCU EM_860 EM_MIPS
   EM_S370 EM_MIPS_RS3_LE EM_PARISC EM_VPP500 EM_SPARC32PLUS EM_960
   EM_PPC EM_PPC64 EM_S390 EM_SPU EM_V800 EM_FR20 EM_RH32 EM_RCE EM_ARM
   EM_FAKE_ALPHA EM_SH EM_SPARCV9 EM_TRICORE EM_ARC EM_H8_300 EM_H8_300H
   EM_H8S EM_H8_500 EM_IA_64 EM_MIPS_X EM_COLDFIRE EM_68HC12 EM_MMA
   EM_PCP EM_NCPU EM_NDR1 EM_STARCORE EM_ME16 EM_ST100 EM_TINYJ EM_X86_64
   EM_PDSP EM_PDP10 EM_PDP11 EM_FX66 EM_ST9PLUS EM_ST7 EM_68HC16
   EM_68HC11 EM_68HC08 EM_68HC05 EM_SVX EM_ST19 EM_VAX EM_CRIS EM_JAVELIN
   EM_FIREPATH EM_ZSP EM_MMIX EM_HUANY EM_PRISM EM_AVR EM_FR30 EM_D10V
   EM_D30V EM_V850 EM_M32R EM_MN10300 EM_MN10200 EM_PJ EM_OPENRISC
   EM_ARC_COMPACT EM_XTENSA EM_VIDEOCORE EM_TMM_GPP EM_NS32K EM_TPC
   EM_SNP1K EM_ST200 EM_IP2K EM_MAX EM_CR EM_F2MC16 EM_MSP430 EM_BLACKFIN
   EM_SE_C33 EM_SEP EM_ARCA EM_UNICORE EM_EXCESS EM_DXP EM_ALTERA_NIOS2
   EM_CRX EM_XGATE EM_C166 EM_M16C EM_DSPIC30F EM_CE EM_M32C EM_TSK3000
   EM_RS08 EM_SHARC EM_ECOG2 EM_SCORE7 EM_DSP24 EM_VIDEOCORE3
   EM_LATTICEMICO32 EM_SE_C17 EM_TI_C6000 EM_TI_C2000 EM_TI_C5500
   EM_TI_ARP32 EM_TI_PRU EM_MMDSP_PLUS EM_CYPRESS_M8C EM_R32C EM_TRIMEDIA
   EM_QDSP6 EM_8051 EM_STXP7X EM_NDS32 EM_ECOG1X EM_MAXQ30 EM_XIMO16
   EM_MANIK EM_CRAYNV2 EM_RX EM_METAG EM_MCST_ELBRUS EM_ECOG16 EM_CR16
   EM_ETPU EM_SLE9X EM_L10M EM_K10M EM_AARCH64 EM_AVR32 EM_STM8 EM_TILE64
   EM_TILEPRO EM_MICROBLAZE EM_CUDA EM_TILEGX EM_CLOUDSHIELD EM_COREA_1ST
   EM_COREA_2ND EM_ARCV2 EM_OPEN8 EM_RL78 EM_VIDEOCORE5 EM_78KOR
   EM_56800EX EM_BA1 EM_BA2 EM_XCORE EM_MCHP_PIC EM_INTELGT EM_KM32
   EM_KMX32 EM_EMX16 EM_EMX8 EM_KVARC EM_CDP EM_COGE EM_COOL EM_NORC
   EM_CSR_KALIMBA EM_Z80 EM_VISIUM EM_FT32 EM_MOXIE EM_AMDGPU EM_RISCV
   EM_BPF EM_CSKY EM_LOONGARCH

   segment-type?
   PT_NULL PT_LOAD PT_DYNAMIC PT_INTERP PT_NOTE PT_SHLIB PT_PHDR PT_TLS
   PT_NUM PT_LOOS PT_GNU_EH_FRAME PT_GNU_STACK PT_GNU_RELRO
   PT_GNU_PROPERTY PT_LOSUNW PT_SUNWBSS PT_SUNWSTACK PT_HISUNW PT_HIOS
   PT_LOPROC PT_HIPROC

   segment-flags?
   PF_X PF_W PF_R

   section-type?
   SHT_NULL SHT_PROGBITS SHT_SYMTAB SHT_STRTAB SHT_RELA SHT_HASH
   SHT_DYNAMIC SHT_NOTE SHT_NOBITS SHT_REL SHT_SHLIB SHT_DYNSYM
   SHT_INIT_ARRAY SHT_FINI_ARRAY SHT_PREINIT_ARRAY SHT_GROUP
   SHT_SYMTAB_SHNDX SHT_NUM SHT_LOOS SHT_HIOS SHT_LOPROC SHT_HIPROC
   SHT_LOUSER SHT_HIUSER

   section-flag?
   SHF_WRITE SHF_ALLOC SHF_EXECINSTR SHF_MERGE SHF_STRINGS
   SHF_INFO_LINK SHF_LINK_ORDER SHF_OS_NONCONFORMING SHF_GROUP SHF_TLS

   dynamic-tag?
   DT_NULL DT_NEEDED DT_PLTRELSZ DT_PLTGOT DT_HASH DT_STRTAB DT_SYMTAB
   DT_RELA DT_RELASZ DT_RELAENT DT_STRSZ DT_SYMENT DT_INIT DT_FINI
   DT_SONAME DT_RPATH DT_SYMBOLIC DT_REL DT_RELSZ DT_RELENT DT_PLTREL
   DT_DEBUG DT_TEXTREL DT_JMPREL DT_BIND_NOW DT_INIT_ARRAY DT_FINI_ARRAY
   DT_INIT_ARRAYSZ DT_FINI_ARRAYSZ DT_RUNPATH DT_FLAGS DT_ENCODING
   DT_PREINIT_ARRAY DT_PREINIT_ARRAYSZ DT_NUM DT_LOOS DT_HIOS DT_LOPROC
   DT_HIPROC

   dynamic-guile-tag?
   DT_LOGUILE DT_GUILE_GC_ROOT DT_GUILE_GC_ROOT_SZ DT_GUILE_ENTRY
   DT_GUILE_VM_VERSION DT_GUILE_FRAME_MAPS DT_HIGUILE

   STB_LOCAL STB_GLOBAL STB_WEAK STB_NUM STB_LOOS STB_GNU_UNIQUE
   STB_HIOS STB_LOPROC STB_HIPROC

   STT_NOTYPE STT_OBJECT STT_FUNC STT_SECTION STT_FILE STT_COMMON
   STT_TLS STT_NUM STT_LOOS STT_GNU_IFUNC STT_HIOS STT_LOPROC STT_HIPROC

   STV_DEFAULT STV_INTERNAL STV_HIDDEN STV_PROTECTED

   NT_PRSTATUS NT_PRFPREG NT_FPREGSET NT_PRPSINFO NT_PRXREG
   NT_TASKSTRUCT NT_PLATFORM NT_AUXV NT_GWINDOWS NT_ASRS NT_PSTATUS
   NT_PSINFO NT_PRCRED NT_UTSNAME NT_LWPSTATUS NT_LWPSINFO NT_PRFPXREG
   NT_SIGINFO NT_FILE NT_PRXFPREG NT_PPC_VMX NT_PPC_SPE NT_PPC_VSX
   NT_PPC_TAR NT_PPC_PPR NT_PPC_DSCR NT_PPC_EBB NT_PPC_PMU NT_PPC_TM_CGPR
   NT_PPC_TM_CFPR NT_PPC_TM_CVMX NT_PPC_TM_CVSX NT_PPC_TM_SPR
   NT_PPC_TM_CTAR NT_PPC_TM_CPPR NT_PPC_TM_CDSCR NT_PPC_PKEY NT_386_TLS
   NT_386_IOPERM NT_X86_XSTATE NT_S390_HIGH_GPRS NT_S390_TIMER
   NT_S390_TODCMP NT_S390_TODPREG NT_S390_CTRS NT_S390_PREFIX
   NT_S390_LAST_BREAK NT_S390_SYSTEM_CALL NT_S390_TDB NT_S390_VXRS_LOW
   NT_S390_VXRS_HIGH NT_S390_GS_CB NT_S390_GS_BC NT_S390_RI_CB NT_ARM_VFP
   NT_ARM_TLS NT_ARM_HW_BREAK NT_ARM_HW_WATCH NT_ARM_SYSTEM_CALL
   NT_ARM_SVE NT_ARM_PAC_MASK NT_ARM_PACA_KEYS NT_ARM_PACG_KEYS
   NT_ARM_TAGGED_ADDR_CTRL NT_ARM_PAC_ENABLED_KEYS NT_VMCOREDD
   NT_MIPS_DSP NT_MIPS_FP_MODE NT_MIPS_MSA NT_VERSION

   PN_XNUM
   SHN_UNDEF
   SHN_LORESERVE
   SHN_LOPROC
   SHN_HIPROC
   SHN_ABS
   SHN_COMMON
   SHN_HIRESERVE
   SHN_XINDEX)

  (import (rnrs arithmetic bitwise (6))
          (rnrs base (6))
          (rnrs bytevectors (6))
          (rnrs lists (6)))

  (define-syntax define-enum
    (syntax-rules ()
      ((define-enum pred (enum val) ...)
       (begin
         (define enum val) ...
         (define (pred x)
           (if (memv x (list val ...)) #t #f))))))


;;; ELF header constants.

  (define-enum elf-identifier?
    (EI_MAG0       0)
    (EI_MAG1       1)
    (EI_MAG2       2)
    (EI_MAG3       3)
    (EI_CLASS      4)
    (EI_DATA       5)
    (EI_VERSION    6)
    (EI_OSABI      7)
    (EI_ABIVERSION 8)
    (EI_PAD        9)
    (EI_NIDENT    16))


  (define ELFMAG0      #x7f)
  (define ELFMAG1      #x45)
  (define ELFMAG2      #x4c)
  (define ELFMAG3      #x46)
  (define (elf-magic? bv)
    (equal? bv #vu8(#x7f #x45 #x4c #x46)))

  (define ELFCLASSNONE    0)       ; Invalid class
  (define-enum elf-class?
    (ELFCLASS32           1)       ; 32-bit objects
    (ELFCLASS64           2))      ; 64-bit objects

  (define ELFDATANONE     0)       ; Invalid data encoding
  (define-enum elf-data?
    (ELFDATA2LSB          1)       ; 2's complement, little endian
    (ELFDATA2MSB          2))      ; 2's complement, big endian


  (define EV_NONE         0)       ; Invalid ELF version
  (define-enum elf-version?
    (EV_CURRENT      1))           ; Current version

  (define-enum elf-osabi?
    (ELFOSABI_NONE           0)    ; UNIX System V ABI
    (ELFOSABI_SYSV           0)    ; Alias
    (ELFOSABI_HPUX           1)    ; HP-UX
    (ELFOSABI_NETBSD         2)    ; NetBSD.
    (ELFOSABI_GNU            3)    ; Object uses GNU ELF extensions.
    (ELFOSABI_SOLARIS        6)    ; Sun Solaris.
    (ELFOSABI_AIX            7)    ; IBM AIX.
    (ELFOSABI_IRIX           8)    ; SGI Irix.
    (ELFOSABI_FREEBSD        9)    ; FreeBSD.
    (ELFOSABI_TRU64          10)   ; Compaq TRU64 UNIX.
    (ELFOSABI_MODESTO        11)   ; Novell Modesto.
    (ELFOSABI_OPENBSD        12)   ; OpenBSD.
    (ELFOSABI_ARM_AEABI      64)   ; ARM EABI
    (ELFOSABI_ARM            97)   ; ARM
    (ELFOSABI_STANDALONE     255)) ; Standalone (embedded) application

  (define (elf-abiversion? x) (zero? x))

  (define (elf-pad? bv)
    (equal? bv (make-bytevector (- EI_NIDENT EI_PAD) 0)))

  (define-enum elf-type?
    (ET_NONE         0)                 ; No file type
    (ET_REL          1)                 ; Relocatable file
    (ET_EXEC         2)                 ; Executable file
    (ET_DYN          3)                 ; Shared object file
    (ET_CORE         4))                ; Core file


;;; Machine types
;;;
;;; Guile currently has a native compiler for: EM_SPARC, EM_386,
;;; EM_MIPS, EM_PPC, EM_PPC64, EM_ARM, EM_SH, EM_SPARCV9, EM_IA_64,
;;; EM_X86_64.
  (define-enum elf-machine?
    (EM_NONE          0)                ; No machine
    (EM_M32           1)                ; AT&T WE 32100
    (EM_SPARC         2)                ; SUN SPARC
    (EM_386           3)                ; Intel 80386
    (EM_68K           4)                ; Motorola m68k family
    (EM_88K           5)                ; Motorola m88k family
    (EM_IAMCU         6)                ; Intel MCU
    (EM_860           7)                ; Intel 80860
    (EM_MIPS          8)                ; MIPS R3000 big-endian
    (EM_S370          9)                ; IBM System/370
    (EM_MIPS_RS3_LE  10)                ; MIPS R3000 little-endian
                                        ; reserved 11-14
    (EM_PARISC       15)                ; HPPA
                                        ; reserved 16
    (EM_VPP500       17)                ; Fujitsu VPP500
    (EM_SPARC32PLUS  18)                ; Sun's "v8plus"
    (EM_960          19)                ; Intel 80960
    (EM_PPC          20)                ; PowerPC
    (EM_PPC64        21)                ; PowerPC 64-bit
    (EM_S390         22)                ; IBM S390
    (EM_SPU          23)                ; IBM SPU/SPC
                                        ; reserved 24-35
    (EM_V800         36)                ; NEC V800 series
    (EM_FR20         37)                ; Fujitsu FR20
    (EM_RH32         38)                ; TRW RH-32
    (EM_RCE          39)                ; Motorola RCE
    (EM_ARM          40)                ; ARM
    (EM_FAKE_ALPHA   41)                ; Digital Alpha
    (EM_SH           42)                ; Hitachi SH
    (EM_SPARCV9      43)                ; SPARC v9 64-bit
    (EM_TRICORE      44)                ; Siemens Tricore
    (EM_ARC          45)                ; Argonaut RISC Core
    (EM_H8_300       46)                ; Hitachi H8/300
    (EM_H8_300H      47)                ; Hitachi H8/300H
    (EM_H8S          48)                ; Hitachi H8S
    (EM_H8_500       49)                ; Hitachi H8/500
    (EM_IA_64        50)                ; Intel Merced
    (EM_MIPS_X       51)                ; Stanford MIPS-X
    (EM_COLDFIRE     52)                ; Motorola Coldfire
    (EM_68HC12       53)                ; Motorola M68HC12
    (EM_MMA          54)                ; Fujitsu MMA Multimedia Accelerator
    (EM_PCP          55)                ; Siemens PCP
    (EM_NCPU         56)                ; Sony nCPU embeeded RISC
    (EM_NDR1         57)                ; Denso NDR1 microprocessor
    (EM_STARCORE     58)                ; Motorola Start*Core processor
    (EM_ME16         59)                ; Toyota ME16 processor
    (EM_ST100        60)                ; STMicroelectronic ST100 processor
    (EM_TINYJ        61)                ; Advanced Logic Corp. Tinyj emb.fam
    (EM_X86_64       62)                ; AMD x86-64 architecture
    (EM_PDSP         63)                ; Sony DSP Processor
    (EM_PDP10        64)                ; Digital PDP-10
    (EM_PDP11        65)                ; Digital PDP-11
    (EM_FX66         66)                ; Siemens FX66 microcontroller
    (EM_ST9PLUS      67)                ; STMicroelectronics ST9+ 8/16 mc
    (EM_ST7          68)                ; STmicroelectronics ST7 8 bit mc
    (EM_68HC16       69)                ; Motorola MC68HC16 microcontroller
    (EM_68HC11       70)                ; Motorola MC68HC11 microcontroller
    (EM_68HC08       71)                ; Motorola MC68HC08 microcontroller
    (EM_68HC05       72)                ; Motorola MC68HC05 microcontroller
    (EM_SVX          73)                ; Silicon Graphics SVx
    (EM_ST19         74)                ; STMicroelectronics ST19 8 bit mc
    (EM_VAX          75)                ; Digital VAX
    (EM_CRIS         76)                ; Axis Communications 32-bit emb.proc
    (EM_JAVELIN      77)                ; Infineon Technologies 32-bit emb.proc
    (EM_FIREPATH     78)                ; Element 14 64-bit DSP Processor
    (EM_ZSP          79)                ; LSI Logic 16-bit DSP Processor
    (EM_MMIX         80)                ; Donald Knuth's educational 64-bit proc
    (EM_HUANY        81)                ; Harvard University machine-independent object files
    (EM_PRISM        82)                ; SiTera Prism
    (EM_AVR          83)                ; Atmel AVR 8-bit microcontroller
    (EM_FR30         84)                ; Fujitsu FR30
    (EM_D10V         85)                ; Mitsubishi D10V
    (EM_D30V         86)                ; Mitsubishi D30V
    (EM_V850         87)                ; NEC v850
    (EM_M32R         88)                ; Mitsubishi M32R
    (EM_MN10300      89)                ; Matsushita MN10300
    (EM_MN10200      90)                ; Matsushita MN10200
    (EM_PJ           91)                ; picoJava
    (EM_OPENRISC     92)                ; OpenRISC 32-bit embedded processor
    (EM_ARC_COMPACT  93)                ; ARC International ARCompact
    (EM_XTENSA       94)                ; Tensilica Xtensa Architecture
    (EM_VIDEOCORE    95)                ; Alphamosaic VideoCore
    (EM_TMM_GPP      96)                ; Thompson Multimedia General Purpose Proc
    (EM_NS32K        97)                ; National Semi. 32000
    (EM_TPC          98)                ; Tenor Network TPC
    (EM_SNP1K        99)                ; Trebia SNP 1000
    (EM_ST200        100)               ; STMicroelectronics ST200
    (EM_IP2K         101)               ; Ubicom IP2xxx
    (EM_MAX          102)               ; MAX processor
    (EM_CR           103)               ; National Semi. CompactRISC
    (EM_F2MC16       104)               ; Fujitsu F2MC16
    (EM_MSP430       105)               ; Texas Instruments msp430
    (EM_BLACKFIN     106)               ; Analog Devices Blackfin DSP
    (EM_SE_C33       107)               ; Seiko Epson S1C33 family
    (EM_SEP          108)               ; Sharp embedded microprocessor
    (EM_ARCA         109)               ; Arca RISC
    (EM_UNICORE      110)               ; PKU-Unity & MPRC Peking Uni. mc series
    (EM_EXCESS       111)               ; eXcess configurable cpu
    (EM_DXP          112)               ; Icera Semi. Deep Execution Processor
    (EM_ALTERA_NIOS2 113)               ; Altera Nios II
    (EM_CRX          114)               ; National Semi. CompactRISC CRX
    (EM_XGATE        115)               ; Motorola XGATE
    (EM_C166         116)               ; Infineon C16x/XC16x
    (EM_M16C         117)               ; Renesas M16C
    (EM_DSPIC30F     118)               ; Microchip Technology dsPIC30F
    (EM_CE           119)               ; Freescale Communication Engine RISC
    (EM_M32C         120)               ; Renesas M32C
                                        ; reserved 121-130
    (EM_TSK3000      131)               ; Altium TSK3000
    (EM_RS08         132)               ; Freescale RS08
    (EM_SHARC        133)               ; Analog Devices SHARC family
    (EM_ECOG2        134)               ; Cyan Technology eCOG2
    (EM_SCORE7       135)               ; Sunplus S+core7 RISC
    (EM_DSP24        136)               ; New Japan Radio (NJR) 24-bit DSP
    (EM_VIDEOCORE3   137)               ; Broadcom VideoCore III
    (EM_LATTICEMICO32 138)              ; RISC for Lattice FPGA
    (EM_SE_C17       139)               ; Seiko Epson C17
    (EM_TI_C6000     140)               ; Texas Instruments TMS320C6000 DSP
    (EM_TI_C2000     141)               ; Texas Instruments TMS320C2000 DSP
    (EM_TI_C5500     142)               ; Texas Instruments TMS320C55x DSP
    (EM_TI_ARP32     143)               ; Texas Instruments App. Specific RISC
    (EM_TI_PRU       144)               ; Texas Instruments Prog. Realtime Unit
                                        ; reserved 145-159
    (EM_MMDSP_PLUS   160)               ; STMicroelectronics 64bit VLIW DSP
    (EM_CYPRESS_M8C  161)               ; Cypress M8C
    (EM_R32C         162)               ; Renesas R32C
    (EM_TRIMEDIA     163)               ; NXP Semi. TriMedia
    (EM_QDSP6        164)               ; QUALCOMM DSP6
    (EM_8051         165)               ; Intel 8051 and variants
    (EM_STXP7X       166)               ; STMicroelectronics STxP7x
    (EM_NDS32        167)               ; Andes Tech. compact code emb. RISC
    (EM_ECOG1X       168)               ; Cyan Technology eCOG1X
    (EM_MAXQ30       169)               ; Dallas Semi. MAXQ30 mc
    (EM_XIMO16       170)               ; New Japan Radio (NJR) 16-bit DSP
    (EM_MANIK        171)               ; M2000 Reconfigurable RISC
    (EM_CRAYNV2      172)               ; Cray NV2 vector architecture
    (EM_RX           173)               ; Renesas RX
    (EM_METAG        174)               ; Imagination Tech. META
    (EM_MCST_ELBRUS  175)               ; MCST Elbrus
    (EM_ECOG16       176)               ; Cyan Technology eCOG16
    (EM_CR16         177)               ; National Semi. CompactRISC CR16
    (EM_ETPU         178)               ; Freescale Extended Time Processing Unit
    (EM_SLE9X        179)               ; Infineon Tech. SLE9X
    (EM_L10M         180)               ; Intel L10M
    (EM_K10M         181)               ; Intel K10M
                                        ; reserved 182
    (EM_AARCH64      183)               ; ARM AARCH64
                                        ; reserved 184
    (EM_AVR32        185)               ; Amtel 32-bit microprocessor
    (EM_STM8         186)               ; STMicroelectronics STM8
    (EM_TILE64       187)               ; Tilera TILE64
    (EM_TILEPRO      188)               ; Tilera TILEPro
    (EM_MICROBLAZE   189)               ; Xilinx MicroBlaze
    (EM_CUDA         190)               ; NVIDIA CUDA
    (EM_TILEGX       191)               ; Tilera TILE-Gx
    (EM_CLOUDSHIELD  192)               ; CloudShield
    (EM_COREA_1ST    193)               ; KIPO-KAIST Core-A 1st gen.
    (EM_COREA_2ND    194)               ; KIPO-KAIST Core-A 2nd gen.
    (EM_ARCV2        195)               ; Synopsys ARCv2 ISA.
    (EM_OPEN8        196)               ; Open8 RISC
    (EM_RL78         197)               ; Renesas RL78
    (EM_VIDEOCORE5   198)               ; Broadcom VideoCore V
    (EM_78KOR        199)               ; Renesas 78KOR
    (EM_56800EX      200)               ; Freescale 56800EX DSC
    (EM_BA1          201)               ; Beyond BA1
    (EM_BA2          202)               ; Beyond BA2
    (EM_XCORE        203)               ; XMOS xCORE
    (EM_MCHP_PIC     204)               ; Microchip 8-bit PIC(r)
    (EM_INTELGT      205)               ; Intel Graphics Technology
                                        ; reserved 206-209
    (EM_KM32         210)               ; KM211 KM32
    (EM_KMX32        211)               ; KM211 KMX32
    (EM_EMX16        212)               ; KM211 KMX16
    (EM_EMX8         213)               ; KM211 KMX8
    (EM_KVARC        214)               ; KM211 KVARC
    (EM_CDP          215)               ; Paneve CDP
    (EM_COGE         216)               ; Cognitive Smart Memory Processor
    (EM_COOL         217)               ; Bluechip CoolEngine
    (EM_NORC         218)               ; Nanoradio Optimized RISC
    (EM_CSR_KALIMBA  219)               ; CSR Kalimba
    (EM_Z80          220)               ; Zilog Z80
    (EM_VISIUM       221)               ; Controls and Data Services VISIUMcore
    (EM_FT32         222)               ; FTDI Chip FT32
    (EM_MOXIE        223)               ; Moxie processor
    (EM_AMDGPU       224)               ; AMD GPU
                                        ; reserved 225-242
    (EM_RISCV        243)               ; RISC-V

    (EM_BPF          247)               ; Linux BPF -- in-kernel virtual machine
    (EM_CSKY         252)               ; C-SKY
    (EM_LOONGARCH    258))              ; LoongArch


;;; Segment types
  (define PT_NULL        0)           ; Program header table entry unused
  (define PT_LOAD        1)           ; Loadable program segment
  (define PT_DYNAMIC     2)           ; Dynamic linking information
  (define PT_INTERP      3)           ; Program interpreter
  (define PT_NOTE        4)           ; Auxiliary information
  (define PT_SHLIB       5)           ; Reserved
  (define PT_PHDR        6)           ; Entry for header table itself
  (define PT_TLS         7)           ; Thread-local storage segment
  (define    PT_NUM      8)           ; Number of defined types
  (define PT_LOOS         #x60000000) ; Start of OS-specific
  (define PT_GNU_EH_FRAME #x6474e550) ; GCC .eh_frame_hdr segment
  (define PT_GNU_STACK    #x6474e551) ; Indicates stack executability
  (define PT_GNU_RELRO    #x6474e552) ; Read-only after relocation
  (define PT_GNU_PROPERTY #x6474e553) ; GNU property
  (define PT_LOSUNW       #x6ffffffa)
  (define PT_SUNWBSS      #x6ffffffa) ; Sun Specific segment
  (define PT_SUNWSTACK    #x6ffffffb) ; Stack segment
  (define PT_HISUNW       #x6fffffff)
  (define PT_HIOS         #x6fffffff) ; End of OS-specific
  (define PT_LOPROC       #x70000000) ; Start of processor-specific
  (define PT_HIPROC       #x7fffffff) ; End of processor-specific

  (define (segment-type? x)
    (or (< -1 x PT_NUM)
        (<= PT_LOOS   x PT_HIOS)
        (<= PT_LOPROC x PT_HIPROC)))


;;; Segment flags

  (define PF_X            1)            ; Segment is executable
  (define PF_W            2)            ; Segment is writable
  (define PF_R            4)            ; Segment is readable

  (define (segment-flags? x)
    (< -1 x 8))


;;; Section Types

  (define-enum section-type?
    (SHT_NULL          0)           ; Section header table entry unused
    (SHT_PROGBITS      1)           ; Program data
    (SHT_SYMTAB        2)           ; Symbol table
    (SHT_STRTAB        3)           ; String table
    (SHT_RELA          4)           ; Relocation entries with addends
    (SHT_HASH          5)           ; Symbol hash table
    (SHT_DYNAMIC       6)           ; Dynamic linking information
    (SHT_NOTE          7)           ; Notes
    (SHT_NOBITS        8)           ; Program space with no data (bss)
    (SHT_REL           9)           ; Relocation entries, no addends
    (SHT_SHLIB         10)          ; Reserved
    (SHT_DYNSYM        11)          ; Dynamic linker symbol table
    (SHT_INIT_ARRAY    14)          ; Array of constructors
    (SHT_FINI_ARRAY    15)          ; Array of destructors
    (SHT_PREINIT_ARRAY 16)          ; Array of pre-constructors
    (SHT_GROUP         17)          ; Section group
    (SHT_SYMTAB_SHNDX  18)          ; Extended section indeces
    (SHT_NUM           19)          ; Number of defined types.
    (SHT_LOOS          #x60000000)  ; Start OS-specific.
    (SHT_HIOS          #x6fffffff)  ; End OS-specific type
    (SHT_LOPROC        #x70000000)  ; Start of processor-specific
    (SHT_HIPROC        #x7fffffff)  ; End of processor-specific
    (SHT_LOUSER        #x80000000)  ; Start of application-specific
    (SHT_HIUSER        #x8fffffff)) ; End of application-specific

;;; Section Flags
  (define-enum section-flag?
    (SHF_WRITE            (bitwise-arithmetic-shift 1  0))  ; Writable
    (SHF_ALLOC            (bitwise-arithmetic-shift 1  1))  ; Occupies memory during execution
    (SHF_EXECINSTR        (bitwise-arithmetic-shift 1  2))  ; Executable
    (SHF_MERGE            (bitwise-arithmetic-shift 1  4))  ; Might be merged
    (SHF_STRINGS          (bitwise-arithmetic-shift 1  5))  ; Contains nul-terminated strings
    (SHF_INFO_LINK        (bitwise-arithmetic-shift 1  6))  ; `sh_info' contains SHT index
    (SHF_LINK_ORDER       (bitwise-arithmetic-shift 1  7))  ; Preserve order after combining
    (SHF_OS_NONCONFORMING (bitwise-arithmetic-shift 1  8))  ; Non-standard OS specific handling required
    (SHF_GROUP            (bitwise-arithmetic-shift 1  9))  ; Section is member of a group.
    (SHF_TLS              (bitwise-arithmetic-shift 1 10))) ; Section hold thread-local data.


;;; Dynamic entry types.

  (define-enum dynamic-tag?
    (DT_NULL              0)          ; Marks end of dynamic section
    (DT_NEEDED            1)          ; Name of needed library
    (DT_PLTRELSZ          2)          ; Size in bytes of PLT relocs
    (DT_PLTGOT            3)          ; Processor defined value
    (DT_HASH              4)          ; Address of symbol hash table
    (DT_STRTAB            5)          ; Address of string table
    (DT_SYMTAB            6)          ; Address of symbol table
    (DT_RELA              7)          ; Address of Rela relocs
    (DT_RELASZ            8)          ; Total size of Rela relocs
    (DT_RELAENT           9)          ; Size of one Rela reloc
    (DT_STRSZ            10)          ; Size of string table
    (DT_SYMENT           11)          ; Size of one symbol table entry
    (DT_INIT             12)          ; Address of init function
    (DT_FINI             13)          ; Address of termination function
    (DT_SONAME           14)          ; Name of shared object
    (DT_RPATH            15)          ; Library search path (deprecated)
    (DT_SYMBOLIC         16)          ; Start symbol search here
    (DT_REL              17)          ; Address of Rel relocs
    (DT_RELSZ            18)          ; Total size of Rel relocs
    (DT_RELENT           19)          ; Size of one Rel reloc
    (DT_PLTREL           20)          ; Type of reloc in PLT
    (DT_DEBUG            21)          ; For debugging ; unspecified
    (DT_TEXTREL          22)          ; Reloc might modify .text
    (DT_JMPREL           23)          ; Address of PLT relocs
    (DT_BIND_NOW         24)          ; Process relocations of object
    (DT_INIT_ARRAY       25)          ; Array with addresses of init fct
    (DT_FINI_ARRAY       26)          ; Array with addresses of fini fct
    (DT_INIT_ARRAYSZ     27)          ; Size in bytes of DT_INIT_ARRAY
    (DT_FINI_ARRAYSZ     28)          ; Size in bytes of DT_FINI_ARRAY
    (DT_RUNPATH          29)          ; Library search path
    (DT_FLAGS            30)          ; Flags for the object being loaded
    (DT_ENCODING         32)          ; Start of encoded range
    (DT_PREINIT_ARRAY    32)          ; Array with addresses of preinit fc
    (DT_PREINIT_ARRAYSZ  33)          ; size in bytes of DT_PREINIT_ARRAY
    (DT_NUM              34)          ; Number used
    (DT_LOOS     #x6000000d)          ; Start of OS-specific
    (DT_HIOS     #x6ffff000)          ; End of OS-specific
    (DT_LOPROC   #x70000000)          ; Start of processor-specific
    (DT_HIPROC   #x7fffffff))         ; End of processor-specific

;;; The DT_GUILE types are non-standard and refer to GNU Guile.

  (define-enum dynamic-guile-tag?
    (DT_LOGUILE          #x37146000)    ; Start of Guile-specific
    (DT_GUILE_GC_ROOT    #x37146000)    ; Offset of GC roots
    (DT_GUILE_GC_ROOT_SZ #x37146001)    ; Size in machine words of GC roots
    (DT_GUILE_ENTRY      #x37146002)    ; Address of entry thunk
    (DT_GUILE_VM_VERSION #x37146003)    ; Bytecode version
    (DT_GUILE_FRAME_MAPS #x37146004)    ; Offset of .guile.frame-maps
    (DT_HIGUILE          #x37146fff))   ; End of Guile-specific


;;; Legal values for note segment descriptor types for core files.

(define NT_PRSTATUS	1)		; Contains copy of prstatus struct
(define NT_PRFPREG	2)		; Contains copy of fpregset struct.
(define NT_FPREGSET	2)		; Contains copy of fpregset struct
(define NT_PRPSINFO	3)		; Contains copy of prpsinfo struct
(define NT_PRXREG	4)		; Contains copy of prxregset struct
(define NT_TASKSTRUCT	4)		; Contains copy of task structure
(define NT_PLATFORM	5)		; String from sysinfo(SI_PLATFORM)
(define NT_AUXV		6)		; Contains copy of auxv array
(define NT_GWINDOWS	7)		; Contains copy of gwindows struct
(define NT_ASRS		8)		; Contains copy of asrset struct
(define NT_PSTATUS	10)		; Contains copy of pstatus struct
(define NT_PSINFO	13)		; Contains copy of psinfo struct
(define NT_PRCRED	14)		; Contains copy of prcred struct
(define NT_UTSNAME	15)		; Contains copy of utsname struct
(define NT_LWPSTATUS	16)		; Contains copy of lwpstatus struct
(define NT_LWPSINFO	17)		; Contains copy of lwpinfo struct
(define NT_PRFPXREG	20)		; Contains copy of fprxregset struct
(define NT_SIGINFO	#x53494749)	; Contains copy of siginfo_t, size might increase
(define NT_FILE		#x46494c45)	; Contains information about mapped files
(define NT_PRXFPREG	#x46e62b7f)	; Contains copy of user_fxsr_struct
(define NT_PPC_VMX	#x100)		; PowerPC Altivec/VMX registers
(define NT_PPC_SPE	#x101)		; PowerPC SPE/EVR registers
(define NT_PPC_VSX	#x102)		; PowerPC VSX registers
(define NT_PPC_TAR	#x103)		; Target Address Register
(define NT_PPC_PPR	#x104)		; Program Priority Register
(define NT_PPC_DSCR	#x105)		; Data Stream Control Register
(define NT_PPC_EBB	#x106)		; Event Based Branch Registers
(define NT_PPC_PMU	#x107)		; Performance Monitor Registers
(define NT_PPC_TM_CGPR	#x108)		; TM checkpointed GPR Registers
(define NT_PPC_TM_CFPR	#x109)		; TM checkpointed FPR Registers
(define NT_PPC_TM_CVMX	#x10a)		; TM checkpointed VMX Registers
(define NT_PPC_TM_CVSX	#x10b)		; TM checkpointed VSX Registers
(define NT_PPC_TM_SPR	#x10c)		; TM Special Purpose Registers
(define NT_PPC_TM_CTAR	#x10d)		; TM checkpointed Target Address Register
(define NT_PPC_TM_CPPR	#x10e)		; TM checkpointed Program Priority Register
(define NT_PPC_TM_CDSCR	#x10f)		; TM checkpointed Data Stream Control Register
(define NT_PPC_PKEY	#x110)		; Memory Protection Keys registers.
(define NT_386_TLS	#x200)		; i386 TLS slots (struct user_desc)
(define NT_386_IOPERM	#x201)		; x86 io permission bitmap (1=deny)
(define NT_X86_XSTATE	#x202)		; x86 extended state using xsave
(define NT_S390_HIGH_GPRS	#x300)	; s390 upper register halves
(define NT_S390_TIMER	#x301)		; s390 timer register
(define NT_S390_TODCMP	#x302)		; s390 TOD clock comparator register
(define NT_S390_TODPREG	#x303)		; s390 TOD programmable register
(define NT_S390_CTRS	#x304)		; s390 control registers
(define NT_S390_PREFIX	#x305)		; s390 prefix register
(define NT_S390_LAST_BREAK	#x306)	; s390 breaking event address
(define NT_S390_SYSTEM_CALL	#x307)	; s390 system call restart data
(define NT_S390_TDB	#x308)		; s390 transaction diagnostic block
(define NT_S390_VXRS_LOW	#x309)	; s390 vector registers 0-15 upper half.
(define NT_S390_VXRS_HIGH	#x30a)	; s390 vector registers 16-31.
(define NT_S390_GS_CB	#x30b)		; s390 guarded storage registers.
(define NT_S390_GS_BC	#x30c)		; s390 guarded storage broadcast control block.
(define NT_S390_RI_CB	#x30d)		; s390 runtime instrumentation.
(define NT_ARM_VFP	#x400)		; ARM VFP/NEON registers
(define NT_ARM_TLS	#x401)		; ARM TLS register
(define NT_ARM_HW_BREAK	#x402)		; ARM hardware breakpoint registers
(define NT_ARM_HW_WATCH	#x403)		; ARM hardware watchpoint registers
(define NT_ARM_SYSTEM_CALL	#x404)	; ARM system call number
(define NT_ARM_SVE	#x405)		; ARM Scalable Vector Extension registers
(define NT_ARM_PAC_MASK	#x406)		; ARM pointer authentication code masks.
(define NT_ARM_PACA_KEYS	#x407)	; ARM pointer authentication address keys.
(define NT_ARM_PACG_KEYS	#x408)	; ARM pointer authentication generic key.
(define NT_ARM_TAGGED_ADDR_CTRL	#x409)	; AArch64 tagged address control.
(define NT_ARM_PAC_ENABLED_KEYS	#x40a)	; AArch64 pointer authentication enabled keys.
(define NT_VMCOREDD	#x700)		; Vmcore Device Dump Note.
(define NT_MIPS_DSP	#x800)		; MIPS DSP ASE registers.
(define NT_MIPS_FP_MODE	#x801)		; MIPS floating-point mode.
(define NT_MIPS_MSA	#x802)		; MIPS SIMD registers.

;;; Legal values for the note segment descriptor types for object files.

(define NT_VERSION	1)		; Contains a version string.


;;; Auxiliary
  (define PN_XNUM       #xffff)


;;; Special section header indices
  (define SHN_UNDEF             0)
  (define SHN_LORESERVE    #xff00)  ; Start of reserved indices
  (define SHN_LOPROC       #xff00)  ; Start of processor-specific
  (define SHN_HIPROC       #xff1f)  ; End of processor-specific
  (define SHN_ABS          #xfff1)  ; Associated symbol is absolute
  (define SHN_COMMON       #xfff2)  ; Associated symbol is common
  (define SHN_HIRESERVE    #xffff)  ; End of reserved indices
  (define SHN_XINDEX       #xffff)


;;; Legal values for ST_BIND subfield of st_info (symbol binding).
  (define STB_LOCAL       0)        ; Local symbol
  (define STB_GLOBAL      1)        ; Global symbol
  (define STB_WEAK        2)        ; Weak symbol
  (define STB_NUM         3)        ; Number of defined types.
  (define STB_LOOS       10)        ; Start of OS-specific
  (define STB_GNU_UNIQUE 10)        ; Unique symbol.
  (define STB_HIOS       12)        ; End of OS-specific
  (define STB_LOPROC     13)        ; Start of processor-specific
  (define STB_HIPROC     15)        ; End of processor-specific

;;; Legal values for ST_TYPE subfield of st_info (symbol type).
  (define STT_NOTYPE     0)         ; Symbol type is unspecified
  (define STT_OBJECT     1)         ; Symbol is a data object
  (define STT_FUNC       2)         ; Symbol is a code object
  (define STT_SECTION    3)         ; Symbol associated with a section
  (define STT_FILE       4)         ; Symbol's name is file name
  (define STT_COMMON     5)         ; Symbol is a common data object
  (define STT_TLS        6)         ; Symbol is thread-local data objec
  (define STT_NUM        7)         ; Number of defined types.
  (define STT_LOOS      10)         ; Start of OS-specific
  (define STT_GNU_IFUNC 10)         ; Symbol is indirect code object
  (define STT_HIOS      12)         ; End of OS-specific
  (define STT_LOPROC    13)         ; Start of processor-specific
  (define STT_HIPROC    15)         ; End of processor-specific

;;; Symbol visibility specification encoded in the st_other field.
  (define STV_DEFAULT   0)          ; Default symbol visibility rules
  (define STV_INTERNAL  1)          ; Processor specific hidden class
  (define STV_HIDDEN    2)          ; Sym unavailable in other modules
  (define STV_PROTECTED 3)          ; Not preemptible, not exported
  )
