;; -*- mode: scheme; coding: utf-8 -*-
;; Copyright (c) 2023 
;; SPDX-License-Identifier: MIT
#!r6rs

(library (vterm common)
  (export
    make-vterm-pos
    vterm-pos?
    vterm-pos=?
    vterm-pos-row
    vterm-pos-row-set!
    vterm-pos-col
    vterm-pos-col-set!

    make-vterm-rect
    vterm-rect?
    vterm-rect-start-row
    vterm-rect-start-row-set!
    vterm-rect-start-col
    vterm-rect-start-col-set!
    vterm-rect-end-row
    vterm-rect-end-row-set!
    vterm-rect-end-col
    vterm-rect-end-col-set!
    vterm-rect-copy

    vterm-rect-expand
    vterm-scroll-rect)
  (import (rnrs)
          (text-mode console model))

(define-record-type vterm-pos
  (fields
    (mutable row)
    (mutable col)))

(define (vterm-pos=? pos1 pos2)
  (and (= (vterm-pos-col pos1)
          (vterm-pos-col pos2))
       (= (vterm-pos-row pos1)
          (vterm-pos-row pos2))))

(define-record-type vterm-rect
  (fields
    (mutable start-row)
    (mutable start-col)
    (mutable end-row)
    (mutable end-col)))

(define (vterm-rect-copy r)
  (make-vterm-rect (vterm-rect-start-row r) (vterm-rect-start-col r)
                   (vterm-rect-end-row r) (vterm-rect-end-col r)))

(define (vterm-rect-expand dst src)
  (when (fx>? (vterm-rect-start-row dst) (vterm-rect-start-row src))
    (vterm-rect-start-row-set! dst (vterm-rect-start-row src)))
  (when (fx>? (vterm-rect-start-col dst) (vterm-rect-start-col src))
    (vterm-rect-start-col-set! dst (vterm-rect-start-col src)))
  (when (fx<? (vterm-rect-end-row dst) (vterm-rect-end-row src))
    (vterm-rect-end-row-set! dst (vterm-rect-end-row src)))
  (when (fx<? (vterm-rect-end-col dst) (vterm-rect-end-col src))
    (vterm-rect-end-col-set! dst (vterm-rect-end-col src))))

(define (vterm-scroll-rect r downward rightward move-fn erase-fn)
  (if (or (fx>=? (abs downward) (fx- (vterm-rect-end-row r) (vterm-rect-start-row r)))
          (fx>=? (abs rightward) (fx- (vterm-rect-end-col r) (vterm-rect-start-col r))))
      (erase-fn r #f)
    (let ([src (make-vterm-rect -1 -1 -1 -1)]
          [dst (make-vterm-rect -1 -1 -1 -1)])
      (cond
        ((fx>=? rightward 0)
         ;; dst
         (vterm-rect-start-col-set! dst (vterm-rect-start-col r))
         (vterm-rect-end-col-set! dst (fx- (vterm-rect-end-col r) rightward))
         ;; src
         (vterm-rect-start-col-set! src (fx+ (vterm-rect-start-col r) rightward))
         (vterm-rect-end-col-set! src (vterm-rect-end-col r)))
        (else
          (let ([leftward (fx- rightward)])
            ;; dst
            (vterm-rect-start-col-set! dst (fx+ (vterm-rect-start-col r) leftward))
            (vterm-rect-end-col-set! dst (vterm-rect-end-col r))
            ;; src
            (vterm-rect-start-col-set! src (vterm-rect-start-col r))
            (vterm-rect-end-col-set! src (fx- (vterm-rect-end-col r) leftward)))))
      (cond
        ((fx>=? downward 0)
         ;; dst
         (vterm-rect-start-row-set! dst (vterm-rect-start-row r))
         (vterm-rect-end-row-set! dst (fx- (vterm-rect-end-row r) downward))
         ;; src
         (vterm-rect-start-row-set! src (fx+ (vterm-rect-start-row r) downward))
         (vterm-rect-end-row-set! src (vterm-rect-end-row r)))
        (else
         (let ([upward (fx- downward)])
           ;; dst
           (vterm-rect-start-row-set! dst (fx+ (vterm-rect-start-row r) upward))
           (vterm-rect-end-row-set! dst (vterm-rect-end-row r))
           ;; src
           (vterm-rect-start-row-set! src (vterm-rect-start-row r))
           (vterm-rect-end-row-set! src (fx- (vterm-rect-end-row r) upward)))))

      (when move-fn
        (move-fn src dst))

      (cond
        ((fx>? downward 0)
         (vterm-rect-start-row-set! r (fx- (vterm-rect-end-row r) downward)))
        ((fx<? downward 0)
         (vterm-rect-end-row-set! r (fx- (vterm-rect-start-row r) downward))))

      (cond
        ((fx>? rightward 0)
         (vterm-rect-start-col-set! r (fx- (vterm-rect-end-col r) rightward)))
        ((fx<? rightward 0)
         (vterm-rect-end-col-set! r (fx- (vterm-rect-start-col r) rightward))))

      (erase-fn r #f) )))

)
