;; -*- mode: scheme; coding: utf-8 -*-
;; Copyright (c) 2023 
;; SPDX-License-Identifier: MIT
#!r6rs

(library (vterm screen)
  (export
    make-vterm-cell
    vterm-cell?
    vterm-cell=?
    vterm-cell-char
    vterm-cell-char-set!
    vterm-cell-width
    vterm-cell-width-set!
    vterm-cell-style
    vterm-cell-style-set!
    vterm-cell-set!

    make-vterm-screen
    vterm-screen-resize
    vterm-screen-ref
    vterm-screen-for-each
    vterm-screen-buffer
    vterm-screen-put-glyph
    vterm-screen-chars
    vterm-screen-string
    vterm-screen-erase
    vterm-screen-scroll
    vterm-screen-style-fg
    vterm-screen-style-fg-set!
    vterm-screen-style-bg
    vterm-screen-style-bg-set!
    vterm-screen-style-attr
    vterm-screen-style-attr-set!

    vterm-screen-scroll-buffer-enable!
    vterm-screen-scroll-buffer-enabled?)
  (import (rnrs)
          (vterm utils)
          (vterm common)
          (vterm style)
          (vterm events)
          (vterm state)
          (text-mode unicode))

(define-record-type vterm-cached-event
  (fields damage)
  (protocol
    (lambda (new)
      (lambda ()
        (new
          (make-vterm-damage-event (make-vterm-rect -1 -1 -1 -1))
        )))))

(define-record-type (vterm-cell $make-vterm-cell vterm-cell?)
  (fields (mutable char) (mutable width) (mutable style)))

(define make-vterm-cell
  (case-lambda
    [() ($make-vterm-cell 'null 1 (make-vterm-style))]
    [(char) ($make-vterm-cell char (if (char? char) (char-width char) 0) (make-vterm-style))]
    [(char style) ($make-vterm-cell char (if (char? char) (char-width char) 0) style)]))

(define (vterm-cell=? c1 c2)
  (and (eqv? (vterm-cell-char c1) (vterm-cell-char c2))
       (eqv? (vterm-cell-width c1) (vterm-cell-width c2))
       (vterm-style=? (vterm-cell-style c1) (vterm-cell-style c2))))

(define vterm-cell-set!
  (case-lambda
    [(cell from)
     (vterm-cell-set! cell (vterm-cell-char from)
                           (vterm-cell-width from)
                           (vterm-cell-style from))]

    [(cell char width style)
     (vterm-cell-char-set! cell char)
     (vterm-style-fg-set! (vterm-cell-style cell) (vterm-style-fg style))
     (vterm-style-bg-set! (vterm-cell-style cell) (vterm-style-bg style))
     (vector-copy! (vterm-style-attr style) 0 (vterm-style-attr (vterm-cell-style cell)) 0
                   (vector-length (vterm-style-attr style)))
     (vterm-cell-width-set! cell width)]
))

(define (make-vterm-screen-buffer rows cols style)
  (let ([buf (make-vector (fx* rows cols))])
    (do ([i 0 (fx+ i 1)]) ((fx=? i (vector-length buf)))
      (let ([cell (make-vterm-cell)])
        (vterm-cell-set! cell 'null 1 style)
        (vector-set! buf i cell)))
    buf))

(define (copy-screen-buffer scr old new-rows new-cols)
  (define (cell-idx row col cols)
    (fx+ (fx* cols row) col))

  (let ([old-rows (vterm-screen-rows scr)]
        [old-cols (vterm-screen-cols scr)]
        [buf (make-vector (fx* new-rows new-cols))])
    (do ([row 0 (fx+ row 1)]) ((fx=? row new-rows))
      (do ([col 0 (fx+ col 1)]) ((fx=? col new-cols))
        (if (and (fx<? row old-rows) (fx<? col old-cols))
            (let ([old-i (cell-idx row col old-cols)]
                  [new-i (cell-idx row col new-cols)])
              (vector-set! buf new-i (vector-ref old old-i)))
          (let ([cell (make-vterm-cell)])
            (vterm-cell-set! cell 'null 1 (vterm-screen-style scr))
            (vector-set! buf (cell-idx row col new-cols) cell)))))
      buf))

(define (vterm-screen-alternative? scr)
  (eqv? (vterm-screen-buffer scr) (vterm-screen-buffer-alternative scr)))

(define-record-type (vterm-screen $make-vterm-screen vterm-screen?)
  (fields
    (mutable rows) (mutable cols)
    (mutable buffer-primary)
    (mutable buffer-alternative)
    (mutable buffer)
    (mutable style)
    (mutable damage-type)
    (mutable damaged)
    (mutable pending-scroll)
    handler
    state
    cached-event
    (mutable scroll-buffer))
    (protocol
      (lambda (new)
        (lambda (state handler)
          (let ([buf (make-vterm-screen-buffer (vterm-state-rows state)
                                               (vterm-state-cols state)
                                               (vterm-state-pen state))])
            (new (vterm-state-rows state) ;; rows
                 (vterm-state-cols state) ;; cols
                 buf ;; primary buffer
                 #f  ;; alternative buffer
                 buf ;; buffer
                 (vterm-state-pen state) ;; style
                 'cell ;; damage-type
                 (make-vterm-rect -1 -1 -1 -1) ;; damaged
                 (make-vterm-rect -1 -1 -1 -1) ;; pending-scroll
                 handler
                 state
                 (make-vterm-cached-event) ;; cached-event
                 #f ;; scroll-buffer
                 ))
        ))))

#;(define (vterm-screen-rows scr)
  (vterm-state-rows (vterm-screen-state scr)))

#;(define (vterm-screen-cols scr)
  (vterm-state-cols (vterm-screen-state scr)))

(define (vterm-screen-at-eol? scr row col)
  (define cell (vterm-screen-ref scr row col))

  (let loop ([col col] [at-eol? (and cell (eqv? (vterm-cell-char cell) 'null))])
    (if (and at-eol? (fx<? col (vterm-screen-cols scr)))
        (let ([cell (vterm-screen-ref scr row col)])
          (loop (fx+ col 1) (eqv? (vterm-cell-char cell) 'null)))
      at-eol?)
))

(define (send-pop-line-event scr evt)
  (send-screen-event scr evt)
  (vterm-pop-line-event-has-more? evt))

(define ($resize scr new-rows new-cols delta)
  (let ([alt-screen? (vterm-screen-alternative? scr)]
        [old-rows (vterm-screen-rows scr)]
        [old-cols (vterm-screen-cols scr)])
    (when (and (not alt-screen?) (fx<? new-rows old-rows))
      ;; find the first blank row after the cursor
      (let ([first-blank-row (fx+ 1 (let loop ([row (fx- old-rows 1)])
                                      (if (or (fx<? row new-rows)
                                              (not (vterm-screen-at-eol? scr row 0))
                                              (eqv? row (vterm-state-cursor-row (vterm-screen-state scr))))
                                          row
                                        (loop (fx- row 1)))
                                      ))])
        (when (fx>? first-blank-row new-rows)
          (vterm-screen-scroll scr (make-vterm-rect 0 0 old-rows old-cols)
                                   (fx- first-blank-row new-rows) 0)
          (vterm-pos-row-set! delta
            (fx- (vterm-pos-row delta) (fx- first-blank-row new-rows))))))

   (vterm-screen-buffer-primary-set! scr
     (copy-screen-buffer scr (vterm-screen-buffer-primary scr) new-rows new-cols))
   (when (vterm-screen-buffer-alternative scr)
     (vterm-screen-buffer-alternative-set! scr
       (copy-screen-buffer scr (vterm-screen-buffer-alternative scr) new-rows new-cols)))

   (vterm-screen-buffer-set! scr (if alt-screen?
                                     (vterm-screen-buffer-alternative scr)
                                   (vterm-screen-buffer-primary scr)))

   (vterm-screen-rows-set! scr new-rows)
   (vterm-screen-cols-set! scr new-cols)

   (when (vterm-screen-scroll-buffer-enabled? scr)
     (vterm-screen-scroll-buffer-set! scr (make-vterm-screen-buffer 1 (vterm-screen-cols scr)
                                          (vterm-screen-style scr))))

   (when (fx>? new-cols old-cols)
     (send-damage-event scr 0 old-cols old-rows new-cols))

   ;; try to pop buffered scrolled lines
   (when (and (fx>? new-rows old-rows)
              (not (vterm-screen-alternative? scr))
              (vterm-screen-scroll-buffer-enabled? scr))
     (let ([rect (make-vterm-rect 0 0 (vterm-screen-rows scr) (vterm-screen-cols scr))]
           [evt (make-vterm-pop-line-event (vterm-screen-scroll-buffer scr) #f)])
       (do ([rows (fx- new-rows old-rows) (fx- rows 1)])
           ((or (fx<=? rows 0) (not (send-pop-line-event scr evt))))
         (vterm-screen-scroll scr rect -1 0)
         ;; fill the cells back
         (let ([cells (vterm-pop-line-event-cells evt)])
           (let loop ([col 0])
             (when (fx<? col (vterm-screen-cols scr))
               (let ([c (vector-ref cells col)])
                 (vterm-cell-set! (vterm-screen-ref scr 0 col) c)
                 (loop (fx+ col (vterm-cell-width c))))
             ))
           (send-damage-event scr (vterm-rect-start-row rect) (vterm-rect-start-col rect)
                                  1 (vterm-rect-end-col rect))
           (vterm-pos-row-set! delta (fx+ (vterm-pos-row delta) 1))
         ))))
))

(define make-vterm-screen
  (case-lambda
    [(state)
     (make-vterm-screen state #f)]
    [(state handler)
     (let ([scr ($make-vterm-screen state handler)])
       (vterm-state-handler-set! state
         (lambda (state evt)
           (cond
             ((vterm-glyph-event? evt)
              (vterm-screen-put-glyph scr (vterm-glyph-event-char evt)
                                          (vterm-glyph-event-width evt)
                                          (vterm-pos-row (vterm-glyph-event-pos evt))
                                          (vterm-pos-col (vterm-glyph-event-pos evt))))
             ((vterm-erase-event? evt)
              (vterm-screen-erase scr (vterm-erase-event-rect evt)
                                      (vterm-erase-event-selective? evt)))

             ((vterm-scroll-event? evt)
              (vterm-screen-scroll scr (vterm-scroll-event-rect evt)
                                       (vterm-scroll-event-downward evt)
                                       (vterm-scroll-event-rightward evt)))

             ((vterm-resize-event? evt)
              ($resize scr (vterm-resize-event-new-rows evt)
                           (vterm-resize-event-new-cols evt)
                           (vterm-resize-event-delta evt)))

             (else (and handler (handler state evt)))
         )))
       scr)
     ]))

(define (vterm-screen-resize scr rows cols)
  (vterm-state-resize (vterm-screen-state scr) rows cols))

(define ($cell-idx scr row col)
  (fx+ (fx* (vterm-screen-cols scr) row)
       col))

(define (vterm-screen-ref scr row col)
  (cond
    ((or (fx<? row 0) (fx>=? row (vterm-screen-rows scr)))
     #f)
    ((or (fx<? col 0) (fx>=? col (vterm-screen-cols scr)))
     #f)
    (else (vector-ref (vterm-screen-buffer scr)
                      ($cell-idx scr row col)))))

(define (vterm-screen-damaged-start-row scr)
  (vterm-rect-start-row (vterm-screen-damaged scr)))

(define (vterm-screen-damaged-start-row-set! scr row)
  (vterm-rect-start-row-set! (vterm-screen-damaged scr) row))

(define (vterm-screen-damaged-start-col scr)
  (vterm-rect-start-col (vterm-screen-damaged scr)))

(define (vterm-screen-damaged-start-col-set! scr col)
  (vterm-rect-start-col-set! (vterm-screen-damaged scr) col))

(define (vterm-screen-damaged-end-row scr)
  (vterm-rect-end-row (vterm-screen-damaged scr)))

(define (vterm-screen-damaged-end-row-set! scr row)
  (vterm-rect-end-row-set! (vterm-screen-damaged scr) row))

(define (vterm-screen-damaged-end-col scr)
  (vterm-rect-end-col (vterm-screen-damaged scr)))

(define (vterm-screen-damaged-end-col-set! scr col)
  (vterm-rect-end-col-set! (vterm-screen-damaged scr) col))

(define (vterm-screen-pending-scroll-start-row scr)
  (vterm-rect-start-row (vterm-screen-pending-scroll scr)))

(define (vterm-screen-pending-scroll-start-row-set! scr row)
  (vterm-rect-start-row-set! (vterm-screen-pending-scroll scr) row))

(define (vterm-screen-pending-scroll-start-col scr)
  (vterm-rect-start-col (vterm-screen-pending-scroll scr)))

(define (vterm-screen-pending-scroll-start-col-set! scr col)
  (vterm-rect-start-col-set! (vterm-screen-pending-scroll scr) col))

(define (vterm-screen-pending-scroll-end-row scr)
  (vterm-rect-end-row (vterm-screen-pending-scroll scr)))

(define (vterm-screen-pending-scroll-end-row-set! scr row)
  (vterm-rect-end-row-set! (vterm-screen-pending-scroll scr) row))

(define (vterm-screen-pending-scroll-end-col scr)
  (vterm-rect-end-col (vterm-screen-pending-scroll scr)))

(define (vterm-screen-pending-scroll-end-col-set! scr col)
  (vterm-rect-end-col-set! (vterm-screen-pending-scroll scr) col))

(define (send-damage-event scr start-row start-col end-row end-col)
  (let ([handler (vterm-screen-handler scr)])
    (when handler
      (let ([evt (vterm-cached-event-damage (vterm-screen-cached-event scr))])
        (vterm-rect-start-row-set! (vterm-damage-event-rect evt) start-row)
        (vterm-rect-start-col-set! (vterm-damage-event-rect evt) start-col)
        (vterm-rect-end-row-set! (vterm-damage-event-rect evt) end-row)
        (vterm-rect-end-col-set! (vterm-damage-event-rect evt) end-col)
        (handler scr evt)))))

(define vterm-screen-put-glyph
  (case-lambda
    [(scr char row col)
     (vterm-screen-put-glyph scr char (char-width char) row col)]

    [(scr char width row col)
     (let ([cell (vterm-screen-ref scr row col)])
       (when cell
         (vterm-cell-set! cell char width (vterm-screen-style scr))
         (send-damage-event scr row col (fx+ row 1) (fx+ col width))))]
))

(define (send-screen-event scr evt)
  (let ([handler (vterm-screen-handler scr)])
    (when handler
      (handler scr evt))))

(define vterm-screen-for-each
  (case-lambda
    [(scr func)
     (vterm-screen-for-each scr 0 0 (vterm-screen-rows scr) (vterm-screen-cols scr)
                                func) ]

    [(scr rect func)
     (vterm-screen-for-each scr (vterm-rect-start-row rect) (vterm-rect-start-col rect)
                                (vterm-rect-end-row rect) (vterm-rect-end-col rect)
                                func) ]

    [(scr start-row start-col end-row end-col func)
     (do ([row start-row (fx+ row 1)]) ((or (fx>=? row end-row) (fx>=? row (vterm-screen-rows scr))))
       (do ([col start-col (fx+ col 1)]) ((or (fx>=? col end-col) (fx>=? col (vterm-screen-cols scr))))
         (let* ([cell (vterm-screen-ref scr row col)])
           (func scr cell row col)))) ]))

(define (vterm-screen-chars scr start-row start-col end-row end-col)
  (let ([chars '()] [padding 0])
    (vterm-screen-for-each scr start-row start-col end-row end-col
      (lambda (scr cell row col)
        (let* ([ch (vterm-cell-char cell)])
          (cond
            ((eqv? 'null ch)
             (set! padding (fx+ padding 1)))
            ((eqv? 'cont ch)
             #f)
            (else
              (let loop ()
                (when (positive? padding)
                  (set! chars (append chars (list (integer->char #x20))))
                  (set! padding (fx- padding 1))
                  (loop)))
              (set! chars (append chars (list ch))))))
        (when (and (fx<? row (fx- end-row 1))
                   (fx=? col (fx- end-col 1)))
          (set! padding 0)
          (set! chars (append chars (list (integer->char #x0a)))))))
    chars))

(define vterm-screen-string
  (case-lambda
    [(scr)
     (vterm-screen-string scr 0 0 (vterm-screen-rows scr) (vterm-screen-cols scr)) ]

    [(scr start-row start-col end-row end-col)
     (list->string
       (vterm-screen-chars scr start-row start-col end-row end-col)) ]))

(define vterm-screen-erase
  (case-lambda
    [(scr rect)
     (vterm-screen-erase scr rect #f) ]
    [(scr rect selective?)
     ($erase-internal scr rect selective?)
     ($erase-user scr rect)]))

(define ($move-cells scr src-row src-col dst-row dst-col num)
  (let ([src-i ($cell-idx scr src-row src-col)]
        [dst-i ($cell-idx scr dst-row dst-col)])
    (vector-move! (vterm-screen-buffer scr)
                  src-i dst-i num
                  (lambda (src src-i dst dst-i)
                    (let ([src-cell (vector-ref src src-i)]
                          [dst-cell (make-vterm-cell)])
                      (vterm-cell-set! dst-cell src-cell)
                      (vector-set! dst dst-i dst-cell))))))

(define ($move-internal scr src dst)
  (when (and (vterm-screen-scroll-buffer-enabled? scr)
             (fx=? (vterm-rect-start-row dst) 0)
             (fx=? (vterm-rect-start-col dst) 0)
             (fx=? (vterm-rect-end-col dst) (vterm-screen-cols scr))
             (not (vterm-screen-alternative? scr)))
    (let ([evt (make-vterm-push-line-event (vterm-screen-scroll-buffer scr))]
          [sc-buf (vterm-screen-scroll-buffer scr)])
      (do ([row 0 (fx+ row 1)]) ((fx=? row (vterm-rect-start-row src)))
        (do ([col 0 (fx+ col 1)]) ((fx=? col (vterm-screen-cols scr)))
          (let ([c (vterm-screen-ref scr row col)])
            (vterm-cell-set! (vector-ref sc-buf col) c)))
        (send-screen-event scr evt))))

  (let* ([cols (fx- (vterm-rect-end-col src) (vterm-rect-start-col src))]
         [downward (fx- (vterm-rect-start-row src) (vterm-rect-start-row dst))]
         [init-row (if (fxnegative? downward)
                       (fx- (vterm-rect-end-row dst) 1)
                       (vterm-rect-start-row dst))]
         [test-row (if (fxnegative? downward)
                       (fx- (vterm-rect-start-row dst) 1)
                       (vterm-rect-end-row dst))]
         [inc-row (if (fxnegative? downward)
                       (fx- 1)
                       1)])
    (do ([row init-row (fx+ row inc-row)]) ((fx=? row test-row))
      ($move-cells scr (fx+ row downward) (vterm-rect-start-col src)
                       row (vterm-rect-start-col dst)
                       cols))))

(define ($erase-internal scr rect selective?)
  (let ([start-row (vterm-rect-start-row rect)]
        [start-col (vterm-rect-start-col rect)]
        [end-row (vterm-rect-end-row rect)]
        [end-col (vterm-rect-end-col rect)])
    (do ([row start-row (fx+ row 1)]) ((or (fx>=? row end-row) (fx>=? row (vterm-screen-rows scr))))
      (do ([col start-col (fx+ col 1)]) ((or (fx>=? col end-col) (fx>=? col (vterm-screen-cols scr))))
        (let ([cell (vterm-screen-ref scr row col)])
          (vterm-cell-set! cell 'null 1 (vterm-screen-style scr))
         )))))

(define ($erase-user scr r)
  (send-damage-event scr (vterm-rect-start-row r) (vterm-rect-start-col r)
                         (vterm-rect-end-row r) (vterm-rect-end-col r)))

(define ($move-user scr src dst)
  (call/cc
    (lambda (return)
      (when (not (eqv? (vterm-screen-damage-type scr) 'scroll))
        (let ([handler (vterm-screen-handler scr)])
          (and handler (eqv? #t (handler scr (make-vterm-move-event src dst)))
                       (return))))
      (send-damage-event scr (vterm-rect-start-row dst) (vterm-rect-start-col dst)
                             (vterm-rect-end-row dst) (vterm-rect-end-col dst)) )))

(define ($scroll-screen scr r downward rightward)
  (call/cc
    (lambda (return)
      (when (not (eqv? (vterm-screen-damage-type scr) 'scroll))
        (vterm-scroll-rect (vterm-rect-copy r) downward rightward
                           ;; move
                           (lambda (src dst)
                             ($move-internal scr src dst))
                           ;; erase
                           (lambda (rect selective?)
                             ($erase-internal scr rect selective?)))

        (vterm-scroll-rect (vterm-rect-copy r) downward rightward
                           ;; move
                           (lambda (src dst)
                             ($move-user scr src dst))
                           ;; erase
                           (lambda (rect selective?)
                             ($erase-user scr rect)))
        (return))
    )))

(define vterm-screen-scroll
  (case-lambda
    [(scr rect downward rightward)
     ($scroll-screen scr rect downward rightward) ]))

(define (vterm-screen-style-fg scr)
  (vterm-style-fg (vterm-screen-style scr)))

(define (vterm-screen-style-fg-set! scr fg)
  (vterm-style-fg-set! (vterm-screen-style scr) fg))

(define (vterm-screen-style-bg scr)
  (vterm-style-bg (vterm-screen-style scr)))

(define (vterm-screen-style-bg-set! scr bg)
  (vterm-style-bg-set! (vterm-screen-style scr) bg))

(define (vterm-screen-style-attr scr)
  (vterm-style-attr (vterm-screen-style scr)))

(define (vterm-screen-style-attr-set! scr attr)
  (vector-copy! attr 0 (vterm-screen-style-attr scr) 0 (vector-length attr)))

(define (vterm-screen-scroll-buffer-enabled? scr)
  (not (eqv? (vterm-screen-scroll-buffer scr) #f)))

(define (vterm-screen-scroll-buffer-enable! scr enable?)
  (if (and enable? (not (vterm-screen-scroll-buffer-enabled? scr)))
      (vterm-screen-scroll-buffer-set! scr (make-vterm-screen-buffer 1 (vterm-screen-cols scr)
                                           (vterm-screen-style scr)))
    (vterm-screen-scroll-buffer-set! scr #f)))

)
