;; -*- mode: scheme; coding: utf-8 -*-
;; Copyright (c) 2023 
;; SPDX-License-Identifier: MIT
#!r6rs

(library (vterm state)
  (export
    make-vterm-state
    vterm-state-reset
    vterm-state-process
    vterm-state-resize
    vterm-state-rows
    vterm-state-cols
    vterm-state-cursor-col
    vterm-state-cursor-row
    vterm-state-decoder-set!
    vterm-state-utf8-enabled?
    vterm-state-utf8-enable!
    vterm-state-handler
    vterm-state-handler-set!
    vterm-state-enable-unhandled-seq-set!

    vterm-state-default-fg
    vterm-state-default-bg

    vterm-state-color-ref
    vterm-state-color-set!

    vterm-state-pen
    vterm-state-pen-fg
    vterm-state-pen-fg-set!
    vterm-state-pen-bg
    vterm-state-pen-bg-set!
    vterm-state-pen-attr-on!
    vterm-state-pen-attr-off!
    vterm-state-pen-attr-set?
    vterm-state-pen-attrs-set?
    vterm-state-pen-reset!

    vterm-state-has-mode?)
  (import
    (rnrs (6))
    (vterm utils)
    (vterm common)
    (vterm events)
    (vterm style)
    (vterm parser)
    (vterm encoding)
    (text-mode unicode)
    (text-mode console model))

(define-record-type vterm-cached-event
  (fields glyph cursor scroll erase style)
  (protocol
    (lambda (new)
      (lambda ()
        (new
          (make-vterm-glyph-event #f #f (make-vterm-pos 0 0))
          (make-vterm-cursor-event (make-vterm-pos 0 0) (make-vterm-pos 0 0) #f)
          (make-vterm-scroll-event (make-vterm-rect -1 -1 -1 -1) 0 0)
          (make-vterm-erase-event (make-vterm-rect -1 -1 -1 -1) #f)
          (make-vterm-style-event #f #f #f)
        )))))

(define ansi-colors
  (vector
    Vterm-Black
    Vterm-Red
    Vterm-Green
    Vterm-Yellow
    Vterm-Blue
    Vterm-Magenta
    Vterm-Cyan
    Vterm-Gray
    ;; high intensity
    Vterm-DarkGray
    Vterm-LightRed
    Vterm-LightGreen
    Vterm-Orange
    Vterm-LightBlue
    Vterm-LightMagenta
    Vterm-LightCyan
    Vterm-White
))

(define-enumeration line-info-state
  (double-width
   double-height-top
   double-height-bottom)
  line-info)

(define-enumeration vterm-mode-state
  (keypad
   cursor
   auto-wrap
   insert
   newline
   cursor-visible
   cursor-blink
   cursor-shape
   alt-screen
   origin
   screen
   left-right-margin
   bracket-paste
   report_focus)
  vterm-mode)

(define-syntax lbound
  (lambda (stx)
    (syntax-case stx ()
      [(_ var min)
       (identifier? #'var)
       #'(when (< var min)
           (set! var min))] )))

(define-syntax ubound
  (lambda (stx)
    (syntax-case stx ()
      [(_ var max)
       (identifier? #'var)
       #'(when (> var max)
           (set! var max))] )))

(define (make-list n x)
  (if (= n 0)
      '()
      (cons x (make-list (- n 1) x))))

(define (make-tab-stops cols)
  (let ([tabs (make-bytevector (fxdiv (+ cols 7) 8))])
    (let loop ([i 0])
      (when (< i cols)
        (if (zero? (mod i 8))
            (set-col-tab-stop tabs i)
            (clear-col-tab-stop tabs i))
        (loop (+ i 1))))
    tabs))

(define (tab-stop-mask col)
  (fxarithmetic-shift-left 1 (bitwise-and col 7)))

(define (tab-stop-idx col)
  (fxarithmetic-shift-right col 3))

(define (set-col-tab-stop tabs col)
  (let* ([idx (tab-stop-idx col)]
         [mask (tab-stop-mask col)]
         [val (bytevector-u8-ref tabs idx)])
    (bytevector-u8-set! tabs idx (bitwise-ior val mask))))

(define (clear-col-tab-stop tabs col)
  (let* ([idx (tab-stop-idx col)]
         [mask (tab-stop-mask col)]
         [val (bytevector-u8-ref tabs idx)])
    (bytevector-u8-set! tabs idx (bitwise-and val (bitwise-not mask)))))

(define (is-col-tab-stop? tabs col)
  (let* ([idx (tab-stop-idx col)]
         [mask (tab-stop-mask col)]
         [val (bytevector-u8-ref tabs idx)])
    (positive? (bitwise-and val mask))))

(define-record-type (vterm-state make-$vterm-state vterm-state?)
  (fields
    (mutable rows)
    (mutable cols)
    (mutable cursor)
    (mutable at-phantom?)
    (mutable scroll-top)
    (mutable scroll-bottom)
    (mutable scroll-left)
    (mutable scroll-right)
    (mutable tab-stops)
    (mutable line-info)
    (mutable mode)
    (mutable gl)
    (mutable gr)
    (mutable gsingle)
    (mutable default-fg)
    (mutable default-bg)
    (mutable colors)
    (mutable pen)
    (mutable parser)
    (mutable handler)
    (immutable cached-event)
    (mutable enable-unhandled-seq))
  (protocol
    (lambda (new)
      (lambda (rows cols handler)
        (new rows cols (make-vterm-pos 0 0)
             #f  ;; at-phantom?
             0 -1 ;; scroll-top, scroll-bottom
             0 -1 ;; scroll-left, scroll-right
             (make-tab-stops cols) ;; tab-stops
             (make-vector rows (line-info)) ;; line-info
             '(auto-wrap) ;; mode
             0  ;; gl-set
             1  ;; gr-set
             0  ;; gsingle-set
             Vterm-Gray ;; default-fg
             Vterm-Black ;; default-bg
             ansi-colors ;; colors
             (make-vterm-style) ;; pen
             #f  ;; parser
             handler ;; handler
             (make-vterm-cached-event) ;; cached-event
             #f ;; enable-unhandled-seq
             )))))

(define (vterm-state-cursor-col state)
  (vterm-pos-col (vterm-state-cursor state)))

(define (vterm-state-cursor-row state)
  (vterm-pos-row (vterm-state-cursor state)))

(define (vterm-state-cursor-col-set! state col)
  (vterm-pos-col-set! (vterm-state-cursor state) col))

(define (vterm-state-cursor-row-set! state row)
  (vterm-pos-row-set! (vterm-state-cursor state) row))

(define (vterm-state-cursor-col-inc! state inc)
  (vterm-pos-col-set! (vterm-state-cursor state)
                      (+ (vterm-state-cursor-col state) inc)))

(define (vterm-state-cursor-row-inc! state inc)
  (vterm-pos-row-set! (vterm-state-cursor state)
                      (+ (vterm-state-cursor-row state) inc)))


(define (vterm-state-row-line-info state row)
  (vector-ref (vterm-state-line-info state) row))

(define vterm-state-row-line-info-set!
  (case-lambda
    [(state row info)
     (vterm-state-row-line-info-set! state row info #f)]
    [(state row info force?)
     (let ([handler (vterm-state-handler state)])
       (send-state-event state (make-vterm-line-info-event info)))
     (vector-set! (vterm-state-line-info state) row info)]))

(define (vterm-state-row-width state row)
  (let ([line-info (vterm-state-row-line-info state row)])
    (if (enum-set-member? 'double-width line-info)
        (fxdiv (vterm-state-cols state) 2)
        (vterm-state-cols state))))

(define (vterm-state-current-row-width state)
  (vterm-state-row-width state (vterm-state-cursor-row state)))

(define (vterm-state-has-mode? state mode)
  (member mode (vterm-state-mode state)))

(define (scroll-region-top state)
  (vterm-state-scroll-top state))

(define (scroll-region-bottom state)
  (if (> (vterm-state-scroll-bottom state) -1)
      (vterm-state-scroll-bottom state)
      (vterm-state-rows state)))

(define (scroll-region-left state)
  (if (vterm-state-has-mode? state 'left-right-margin)
      (vterm-state-scroll-left state)
      0))

(define (scroll-region-right state)
  (if (and (vterm-state-has-mode? state 'left-right-margin)
           (> (vterm-state-scroll-right state) -1))
      (vterm-state-scroll-right state)
      (vterm-state-cols state)))

(define (cursor-in-scroll-region? state)
  (cond
    ((or (fx<? (vterm-state-cursor-row state) (scroll-region-top state))
         (fx>=? (vterm-state-cursor-row state) (scroll-region-bottom state)))
     #f)
    ((or (fx<? (vterm-state-cursor-col state) (scroll-region-left state))
         (fx>=? (vterm-state-cursor-col state) (scroll-region-right state)))
     #f)
    (else #t)))

(define (send-state-event state evt)
  (let ([handler (vterm-state-handler state)])
    (when handler
      (handler state evt))))

(define (send-unhandled-seq-event state seq)
  (when (vterm-state-enable-unhandled-seq state)
    (send-state-event state seq)))

(define (send-scroll-event state start-row end-row start-col end-col downward rightward)
  (when (not (and (fxzero? downward) (fxzero? rightward)))
    (let* ([rows (fx- end-row start-row)]
           [downward (cond
                       ((fx>? downward rows) rows)
                       ((fx<? downward (fx- rows)) (fx- rows))
                       (else downward))]
           [cols (fx- end-col start-col)]
           [rightward (cond
                        ((fx>? rightward cols) cols)
                        ((fx<? rightward (fx- cols)) (fx- cols))
                        (else rightward))]
           [handler (vterm-state-handler state)])
      ;; update lineinfo if full line
      (when (and (fx=? start-col 0) (fx=? end-col (vterm-state-cols state)) (fx=? rightward 0))
        (let ([height (- end-row start-row (abs downward))])
          (if (fx>? downward 0)
              (vector-move! (vterm-state-line-info state)
                            (fx+ start-row downward) start-row
                            height)
              (vector-move! (vterm-state-line-info state)
                            start-row (fx- start-row downward)
                            height))))
      (let ([evt (vterm-cached-event-scroll (vterm-state-cached-event state))])
        (let ([rect (vterm-scroll-event-rect evt)])
          (vterm-rect-start-row-set! rect start-row)
          (vterm-rect-start-col-set! rect start-col)
          (vterm-rect-end-row-set! rect end-row)
          (vterm-rect-end-col-set! rect end-col))
        (vterm-scroll-event-downward-set! evt downward)
        (vterm-scroll-event-rightward-set! evt rightward)
        (send-state-event state evt))
        )))

(define (send-erase-event state start-row end-row start-col end-col selective?)
  (let ([evt (vterm-cached-event-erase (vterm-state-cached-event state))])
    (let ([rect (vterm-erase-event-rect evt)])
      (vterm-rect-start-row-set! rect start-row)
      (vterm-rect-start-col-set! rect start-col)
      (vterm-rect-end-row-set! rect end-row)
      (vterm-rect-end-col-set! rect end-col))
    (vterm-erase-event-selective?-set! evt selective?)
    (send-state-event state evt)))

(define (do-linefeed state)
  (cond
    [(= (vterm-state-cursor-row state)
        (- (scroll-region-bottom state) 1))
     (send-scroll-event state (scroll-region-top state) (scroll-region-bottom state)
                              (scroll-region-left state) (scroll-region-right state)
                              1 0)]
    [(< (vterm-state-cursor-row state)
        (- (vterm-state-rows state) 1))
     (vterm-state-cursor-row-inc! state 1)
    ]))

(define (update-cursor state old-pos clean-phantom?)
  (when (not (vterm-pos=? old-pos (vterm-state-cursor state)))
    (when clean-phantom?
      (vterm-state-at-phantom?-set! state #f))
    (let ([evt (vterm-cached-event-cursor (vterm-state-cached-event state))])
      (vterm-pos-row-set! (vterm-cursor-event-new-pos evt) (vterm-state-cursor-row state))
      (vterm-pos-col-set! (vterm-cursor-event-new-pos evt) (vterm-state-cursor-col state))
      (vterm-pos-row-set! (vterm-cursor-event-old-pos evt) (vterm-pos-row old-pos))
      (vterm-pos-col-set! (vterm-cursor-event-old-pos evt) (vterm-pos-col old-pos))
      (vterm-cursor-event-visible?-set! evt (vterm-state-has-mode? state 'cursor-visible))
      (send-state-event state evt))
))

;; TODO extend with extra pos
(define (put-glyph state char width)
  (let* ([evt (vterm-cached-event-glyph (vterm-state-cached-event state))]
         [pos (vterm-glyph-event-pos evt)])
    (vterm-pos-row-set! pos (vterm-state-cursor-row state))
    (vterm-pos-col-set! pos (vterm-state-cursor-col state))
    (vterm-glyph-event-char-set! evt char)
    (vterm-glyph-event-width-set! evt width)
    (send-state-event state evt)))

(define (handle-text state text)
  (define old-pos (make-vterm-pos (vterm-state-cursor-row state)
                                  (vterm-state-cursor-col state)))
  (for-each
    (lambda (ch)
      (let ([width (char-width ch)])
        (when (or (vterm-state-at-phantom? state)
                  (> (+ width (vterm-state-cursor-col state))
                     (vterm-state-current-row-width state)))
          (do-linefeed state)
          (vterm-state-cursor-col-set! state 0)
          (vterm-state-at-phantom?-set! state #f))
        (when (vterm-state-has-mode? state 'insert)
          (let ([start-row (vterm-state-cursor-row state)]
                [end-row (fx+ (vterm-state-cursor-row state) 1)]
                [start-col (vterm-state-cursor-col state)]
                [end-col (vterm-state-current-row-width state)])
            (send-scroll-event state start-row end-row start-col end-col 0 -1)))
        (put-glyph state ch width)
        (if (>= (+ width (vterm-state-cursor-col state))
                (vterm-state-current-row-width state))
            (when (vterm-state-has-mode? state 'auto-wrap)
              (vterm-state-at-phantom?-set! state #t))
          (vterm-state-cursor-col-inc! state width))))
    (string->list text))
  (update-cursor state old-pos #f))

(define (do-tab state count dir)
  (let loop ([count count] [done? #f])
    (when (and (positive? count) (not done?))
      (cond
        ((fxpositive? dir)
         (if (< (vterm-state-cursor-col state)
                (- (vterm-state-current-row-width state) 1))
             (vterm-state-cursor-col-inc! state 1)
             (set! done? #t)))
        ((fxnegative? dir)
         (if (positive? (vterm-state-cursor-col state))
             (vterm-state-cursor-col-inc! state -1)
             (set! done? #t))))
      (when (is-col-tab-stop?
              (vterm-state-tab-stops state)
              (vterm-state-cursor-col state))
        (set! count (- count 1)))
      (loop count done?))))

(define (handle-ctl state code)
  (define old-pos (make-vterm-pos (vterm-state-cursor-row state)
                                  (vterm-state-cursor-col state)))
  (send-state-event state (make-vterm-control-event code))
  (case code
    ;; bs
    ((#x8)
     (when (positive? (vterm-state-cursor-col state))
       (vterm-state-cursor-col-inc! state -1)))

    ;; ht
    ((#x9)
     (do-tab state 1 +1))

    ;; lf vt ff
    ((#xa #xb #xc)
     (do-linefeed state)
     (when (vterm-state-has-mode? state 'newline)
       (vterm-state-cursor-col-set! state 0)))

    ;; cr
    ((#xd)
     (vterm-state-cursor-col-set! state 0))

    ;; ls1
    ((#xe)
     (vterm-state-gl-set! state 1))

    ;; ls0
    ((#xf)
     (vterm-state-gl-set! state 0))

    ;; ind
    ((#x84)
     (do-linefeed state))

    ;; nel
    ((#x85)
     (do-linefeed state)
     (vterm-state-cursor-col-set! state 0))

    ;; hts
    ((#x88)
     (set-col-tab-stop (vterm-state-tab-stops state)
                       (vterm-state-cursor-col state)))

    ;; ri
    ((#x8d)
     (if (= (vterm-state-cursor-row state)
            (scroll-region-top state))
         (send-scroll-event state (scroll-region-top state) (scroll-region-bottom state)
                                  (scroll-region-left state) (scroll-region-right state)
                                  -1 0)
         (when (positive? (vterm-state-cursor-row state))
           (vterm-state-cursor-row-inc! state -1))))

    ;; ss2
    ((#x8e)
     (vterm-state-gsingle-set! state 2))

    ;; ss3
    ((#x8f)
     (vterm-state-gsingle-set! state 3))

    (else
      (send-unhandled-seq-event state (list 'ctl code)))
  )
  (update-cursor state old-pos #t)
)

(define (csi-arg-or args i def)
  (if (< i (length args))
      (let ([arg (list-ref args i)])
        (if arg
            (list-ref args i)
          def))
    def))

(define (csi-arg-count args i)
  (if (< i (length args))
      (let ([arg (list-ref args i)])
        (if (and arg (positive? arg)) arg 1))
    1))

(define (handle-style state args)
  (define (lookup-color args n-args)
     (case (car args)
       ((2)
        (if (fx<? n-args 3)
            (values #f n-args)
            (let ([args (cdr args)]) ;; TODO rgb color handling
              3
            )))
       ((5)
        (if (or (fx=? n-args 0) (not (car args)))
            (values #f (if (not (fx=? n-args 0)) 1 0))
            (let ([args (cdr args)]) ;; TODO handle XTerm 256-color mode
              (values #f (if (not (fx=? n-args 0)) 1 0))
            )))
         (else (values #f 0))
       ))

  (do ([i 0 (fx+ i 1)]) ((fx>=? i (length args)))
    (let ([arg (list-ref args i)])
      (case arg
        ((#f 0)
         (vterm-state-pen-reset! state))
        ((1)
         (vterm-state-pen-attr-on! state Vterm-Bold))
        ((3)
         (vterm-state-pen-attr-on! state Vterm-Italic))
        ((5)
         (vterm-state-pen-attr-on! state Vterm-Blink))
        ((7)
         (vterm-state-pen-attr-on! state Vterm-Reverse))
        ((9)
         (vterm-state-pen-attr-on! state Vterm-Strike))
        ((22)
         (vterm-state-pen-attr-off! state Vterm-Bold))
        ((23)
         (vterm-state-pen-attr-off! state Vterm-Italic))
        ((24)
         (vterm-state-pen-attr-off! state Vterm-Underline))
        ((25)
         (vterm-state-pen-attr-off! state Vterm-Blink))
        ((27)
         (vterm-state-pen-attr-off! state Vterm-Reverse))
        ((29)
         (vterm-state-pen-attr-off! state Vterm-Strike))
        ((30 31 32 33 34 35 36 37)
         (let ([color (fx+ (fx- arg 30)
                           (if (vterm-state-pen-attr-set? state Vterm-Bold)
                               8 0))])
           (vterm-state-pen-fg-set! state (vector-ref (vterm-state-colors state) color))))
        ((38)
         (if (fx<? (fx- (length args) i) 1)
             (set! i (length args)) ;; break
           (let-values ([(fg n-args) (lookup-color (cdr args) (fx- (fx- (length args) i) 2))])
             (set! i (fx+ i (fx+ n-args 1))))))
        ((39)
         (vterm-state-pen-fg-set! state (vterm-state-default-fg state)))
        ((40 41 42 43 44 45 46 47)
         (let ([color (fx- arg 40)])
           (vterm-state-pen-bg-set! state (vector-ref (vterm-state-colors state) color))))
        ((48)
         (if (fx<? (fx- (length args) i) 1)
             (set! i (length args)) ;; break
           (let-values ([(bg n-args) (lookup-color (cdr args) (fx- (fx- (length args) i) 2))])
             (set! i (fx+ i (fx+ n-args 1))))))
        ((49)
         (vterm-state-pen-bg-set! state (vterm-state-default-bg state)))
      )
)))

(define (vterm-mode-set! state mode enable?)
  (when mode
    (let ([ls (vterm-state-mode state)])
      (set! ls (remove mode ls))
      (when enable?
        (set! ls (append ls (list mode))))
      (vterm-state-mode-set! state ls))))

(define (handle-props state args enable?)
  (define key #f)
  (define val #f)

  (when (not (null? args))
    (case (car args)
      ((1)
       (vterm-mode-set! state 'cursor enable?))

      ((25)
       (set! key 'cursor)
       (set! val (if enable? 'on 'off)))

      (else
        (send-unhandled-seq-event state (list 'dec-mode args enable?)))
    )
    (when key
      (let ([handler (vterm-state-handler state)])
        (send-state-event state (make-vterm-property-event key val))))
  ))

(define (set-mode state arg enable?)
  (let ([mode #f])
    (case arg
      ((4) (set! mode 'insert))
      ((20) (set! mode 'newline)))
    (vterm-mode-set! state mode enable?)
))

(define (handle-csi state cmd args interm)
  (let ([leader (and (not (null? args))
                     (string? (car args))
                     (string-ref (car args) 0))])
    (call/cc
      (lambda (return)
        (when (and leader (not (member leader '(#\? #\>))))
          (return))
        (when (and interm
                   (or (not (= (string-length interm) 1))
                       (not (member (string-ref interm 0)
                                    '(#\space #\" #\$ #\')))))
          (return))
        (let* ([leader-byte (if leader (char->integer leader) 0)]
               [interm-byte (if interm (char->integer (string-ref interm 0)) 0)]
               [interm-char (if interm (string-ref interm 0) #f)]
               [command-byte (char->integer cmd)]
               [cancel-phantom? #t]
               [code command-byte]
               [old-pos (make-vterm-pos (vterm-state-cursor-row state)
                                        (vterm-state-cursor-col state))])
          (cond
            ((eqv? code #x40) ;; ICH
             (when (cursor-in-scroll-region? state)
               (let ([count (csi-arg-count args 0)]
                     [start-row (vterm-state-cursor-row state)]
                     [end-row (fx+ (vterm-state-cursor-row state) 1)]
                     [start-col (vterm-state-cursor-col state)]
                     [end-col (if (vterm-state-has-mode? state 'left-right-margin)
                                  (scroll-region-right state)
                                  (vterm-state-current-row-width state))])
                  (send-scroll-event state start-row end-row start-col end-col 0 (fx- count)))))

            ((eqv? code #x41) ;; CUU
             (let ([count (csi-arg-count args 0)])
               (vterm-state-cursor-row-inc! state (- count))
               (vterm-state-at-phantom?-set! state #f)))

            ((eqv? code #x42) ;; CUD
             (let ([count (csi-arg-count args 0)])
               (vterm-state-cursor-row-inc! state count)
               (vterm-state-at-phantom?-set! state #f)))

            ((eqv? code #x43) ;; CUF
             (let ([count (csi-arg-count args 0)])
               (vterm-state-cursor-col-inc! state count)
               (vterm-state-at-phantom?-set! state #f)))

            ((eqv? code #x44) ;; CUB
             (let ([count (csi-arg-count args 0)])
               (vterm-state-cursor-col-inc! state (- count))
               (vterm-state-at-phantom?-set! state #f)))

            ((eqv? code #x45) ;; CNL
             (let ([count (csi-arg-count args 0)])
               (vterm-state-cursor-col-set! state 0)
               (vterm-state-cursor-row-inc! state count)
               (vterm-state-at-phantom?-set! state #f)))

            ((eqv? code #x46) ;; CPL
             (let ([count (csi-arg-count args 0)])
               (vterm-state-cursor-col-set! state 0)
               (vterm-state-cursor-row-inc! state (- count))
               (vterm-state-at-phantom?-set! state #f)))

            ((eqv? code #x47) ;; CHA
             (let ([col (csi-arg-or args 0 1)])
               (vterm-state-cursor-col-set! state (- col 1))
               (vterm-state-at-phantom?-set! state #f)))

            ((eqv? code #x48) ;; CUP
             (let ([row (csi-arg-or args 0 1)]
                   [col (csi-arg-or args 1 1)])
               (vterm-state-cursor-col-set! state (- col 1))
               (vterm-state-cursor-row-set! state (- row 1))
               (when (vterm-state-has-mode? state 'origin)
                 (vterm-state-cursor-row-inc! state (scroll-region-top state))
                 (vterm-state-cursor-col-inc! state (scroll-region-left state)))
               (vterm-state-at-phantom?-set! state #f)))
            ((eqv? code #x49) ;; CHT
             (let ([count (csi-arg-count args 0)])
               (do-tab state count +1)))

            ;; ED, DECSED
            ((or (eqv? code #x4a) (and (eqv? code #x4a) (eqv? leader #\?)))
             (let* ([selective? (eqv? leader #\?)]
                    [args (if selective? (cdr args) args)]
                    [start-row 0]
                    [end-row 0]
                    [start-col 0]
                    [end-col 0])
               (case (car args)
                 ((#f 0)
                  (set! start-row (vterm-state-cursor-row state))
                  (set! end-row (fx+ (vterm-state-cursor-row state) 1))
                  (set! start-col (vterm-state-cursor-col state))
                  (set! end-col (vterm-state-cols state))
                  (when (fx>? end-col start-col)
                    (send-erase-event state start-row end-row start-col end-col selective?))
                  (set! start-row (fx+ (vterm-state-cursor-row state) 1))
                  (set! end-row (vterm-state-rows state))
                  (set! start-col 0)
                  (do ([row start-row (fx+ row 1)]) ((fx=? row end-row))
                    (vterm-state-row-line-info-set! state row (line-info) 'force))
                  (when (fx>? end-row start-row)
                    (send-erase-event state start-row end-row start-col end-col selective?)))

                 ((1)
                  (set! start-row 0)
                  (set! end-row (vterm-state-cursor-row state))
                  (set! start-col 0)
                  (set! end-col (vterm-state-cols state))
                  (do ([row start-row (fx+ row 1)]) ((fx=? row end-row))
                    (vterm-state-row-line-info-set! state row (line-info) 'force))
                  (when (fx>? end-col start-col)
                    (send-erase-event state start-row end-row start-col end-col selective?))
                  (set! start-row (vterm-state-cursor-row state))
                  (set! end-row (fx+ (vterm-state-cursor-row state) 1))
                  (set! end-col (fx+ (vterm-state-cursor-col state) 1))
                  (when (fx>? end-row start-row)
                    (send-erase-event state start-row end-row start-col end-col selective?)))

                 ((2)
                  (set! start-row 0)
                  (set! end-row (vterm-state-rows state))
                  (set! start-col 0)
                  (set! end-col (vterm-state-cols state))
                  (do ([row start-row (fx+ row 1)]) ((fx=? row end-row))
                    (vterm-state-row-line-info-set! state row (line-info) 'force))
                  (send-erase-event state start-row end-row start-col end-col selective?)))))

            ;; EL, DECSEL
            ((or (eqv? code #x4b) (and (eqv? code #x4b) (eqv? leader #\?)))
             (let* ([selective? (eqv? leader #\?)]
                    [args (if selective? (cdr args) args)] 
                    [start-row (vterm-state-cursor-row state)]
                    [end-row (fx+ (vterm-state-cursor-row state) 1)]
                    [start-col 0]
                    [end-col 0])
               (case (car args)
                 ((#f 0)
                  (set! start-col (vterm-state-cursor-col state))
                  (set! end-col (vterm-state-current-row-width state)))
                 ((1)
                  (set! start-col 0)
                  (set! end-col (fx+ (vterm-state-cursor-col state) 1)))
                 ((2)
                  (set! start-col 0)
                  (set! end-col (vterm-state-current-row-width state))))
               (when (fx>? end-col start-col)
                 (send-erase-event state start-row end-row start-col end-col selective?))
              ))

            ((eqv? code #x4c) ;; IL
             (when (cursor-in-scroll-region? state)
               (let ([count (csi-arg-count args 0)]
                     [start-row (vterm-state-cursor-row state)]
                     [end-row (scroll-region-bottom state)]
                     [start-col (scroll-region-left state)]
                     [end-col (scroll-region-right state)])
                  (send-scroll-event state start-row end-row start-col end-col (fx- count) 0))))

            ((eqv? code #x4d) ;; DL
             (when (cursor-in-scroll-region? state)
               (let ([count (csi-arg-count args 0)]
                     [start-row (vterm-state-cursor-row state)]
                     [end-row (scroll-region-bottom state)]
                     [start-col (scroll-region-left state)]
                     [end-col (scroll-region-right state)])
                  (send-scroll-event state start-row end-row start-col end-col count 0))))

            ((eqv? code #x50) ;; DCH
             (when (cursor-in-scroll-region? state)
               (let ([count (csi-arg-count args 0)]
                     [start-row (vterm-state-cursor-row state)]
                     [end-row (fx+ (vterm-state-cursor-row state) 1)]
                     [start-col (vterm-state-cursor-col state)]
                     [end-col (if (vterm-state-has-mode? state 'left-right-margin)
                                  (scroll-region-right state)
                                  (vterm-state-current-row-width state))])
                  (send-scroll-event state start-row end-row start-col end-col 0 count))))

            ((eqv? code #x53) ;; SU
             (let ([count (csi-arg-count args 0)]
                   [start-row (scroll-region-top state)]
                   [end-row (scroll-region-bottom state)]
                   [start-col (scroll-region-left state)]
                   [end-col (scroll-region-right state)])
                (send-scroll-event state start-row end-row start-col end-col count 0)))

            ((eqv? code #x54) ;; SD
             (let ([count (csi-arg-count args 0)]
                   [start-row (scroll-region-top state)]
                   [end-row (scroll-region-bottom state)]
                   [start-col (scroll-region-left state)]
                   [end-col (scroll-region-right state)])
                (send-scroll-event state start-row end-row start-col end-col (fx- count) 0)))

            ((eqv? code #x58) ;; ECH
             (let* ([count (csi-arg-count args 0)]
                    [start-row (vterm-state-cursor-row state)]
                    [end-row (fx+ (vterm-state-cursor-row state) 1)]
                    [start-col (vterm-state-cursor-col state)]
                    [end-col (fx+ (vterm-state-cursor-col state) count)])
                 (ubound end-col (vterm-state-current-row-width state))
                 (send-erase-event state start-row end-row start-col end-col #f)))

            ((eqv? code #x5a) ;; CBT
             (let ([count (csi-arg-count args 0)])
               (do-tab state count -1)))

            ((eqv? code #x60) ;; HPA
             (let ([col (csi-arg-or args 0 1)])
               (vterm-state-cursor-col-set! state (- col 1))
               (vterm-state-at-phantom?-set! state #f)))

            ((eqv? code #x61) ;; HPR
             (let ([col (csi-arg-count args 0)])
               (vterm-state-cursor-col-inc! state col)
               (vterm-state-at-phantom?-set! state #f)))

            ((eqv? code #x64) ;; VPA
             (let ([row (csi-arg-or args 0 1)])
               (vterm-state-cursor-row-set! state (- row 1))
               (when (vterm-state-has-mode? state 'origin)
                 (vterm-state-cursor-row-inc! state (scroll-region-top state)))
               (vterm-state-at-phantom?-set! state #f)))

            ((eqv? code #x65) ;; VPR
             (let ([count (csi-arg-count args 0)])
               (vterm-state-cursor-row-inc! state count)
               (vterm-state-at-phantom?-set! state #f)))

            ((eqv? code #x66) ;; HVP
             (let ([row (csi-arg-or args 0 1)]
                   [col (csi-arg-or args 1 1)])
               (vterm-state-cursor-col-set! state (- col 1))
               (vterm-state-cursor-row-set! state (- row 1))
               (when (vterm-state-has-mode? state 'origin)
                 (vterm-state-cursor-row-inc! state (scroll-region-top state))
                 (vterm-state-cursor-col-inc! state (scroll-region-left state)))
               (vterm-state-at-phantom?-set! state #f)))

            ((eqv? code #x67) ;; TBC
             (let ([val (csi-arg-or args 0 0)])
               (cond
                 ((eqv? val 0)
                  (clear-col-tab-stop (vterm-state-tab-stops state) (vterm-state-cursor-col state)))
                 ((or (eqv? val 3) (eqv? val 5))
                  (let loop ([i 0])
                    (when (< i val)
                      (clear-col-tab-stop (vterm-state-tab-stops state) i)
                      (loop (fx+ i 1))) )))))

           ;; SM
           ((and (not leader) (eqv? code #x68))
            (set-mode state (car args) #t))

           ;; DEC private mode
           ((and (eqv? #\? leader) (eqv? code #x68))
            (handle-props state (cdr args) #t))
           ((and (eqv? #\? leader) (eqv? code #x6c))
            (handle-props state (cdr args) #f))

            ((eqv? code #x6a) ;; HPB
             (let ([count (csi-arg-count args 0)])
               (vterm-state-cursor-col-inc! state (- count))
               (vterm-state-at-phantom?-set! state #f)))

            ((eqv? code #x6b) ;; VPB
             (let ([count (csi-arg-count args 0)])
               (vterm-state-cursor-row-inc! state (- count))
               (vterm-state-at-phantom?-set! state #f)))

           ;; RM
           ((and (not leader) (eqv? code #x6c))
            (set-mode state (car args) #f))

            ((eqv? code #x6d)
             (handle-style state args))

            ;; DECSTBM
            ((eqv? code #x72)
             (let ([scroll-top (fx- (csi-arg-or args 0 1) 1)]
                   [scroll-bottom (cond
                                    ((fx<? (length args) 2) 1)
                                    ((not (list-ref args 1)) -1)
                                    (else (list-ref args 1))) ])
               (lbound scroll-top 0)
               (ubound scroll-top (vterm-state-rows state))
               (lbound scroll-bottom -1)
               (if (and (eqv? scroll-top 0) (eqv? scroll-bottom (vterm-state-rows state)))
                   (set! scroll-bottom -1)
                   (ubound scroll-bottom (vterm-state-rows state)))
               (vterm-state-scroll-top-set! state scroll-top)
               (vterm-state-scroll-bottom-set! state scroll-bottom)
               (when (fx<=? (scroll-region-bottom state) (scroll-region-top state))
                 (vterm-state-scroll-top-set! state 0)
                 (vterm-state-scroll-bottom-set! state -1))
               (vterm-state-cursor-row-set! state 0)
               (vterm-state-cursor-col-set! state 0)))

            ;; DECSLRM
            ((eqv? code #x73)
             (let ([scroll-left (fx- (csi-arg-or args 0 1) 1)]
                   [scroll-right (cond
                                   ((fx<? (length args) 2) 1)
                                   ((not (list-ref args 1)) -1)
                                   (else (list-ref args 1))) ])
               (lbound scroll-left 0)
               (ubound scroll-left (vterm-state-cols state))
               (lbound scroll-right -1)
               (if (and (eqv? scroll-left 0) (eqv? scroll-right (vterm-state-cols state)))
                   (set! scroll-right -1)
                   (ubound scroll-right (vterm-state-cols state)))
               (vterm-state-scroll-left-set! state scroll-left)
               (vterm-state-scroll-right-set! state scroll-right)
               (when (and (fx>? scroll-right -1) (fx<=? scroll-right scroll-left))
                 (vterm-state-scroll-left-set! state 0)
                 (vterm-state-scroll-right-set! state -1))
               (vterm-state-cursor-row-set! state 0)
               (vterm-state-cursor-col-set! state 0)))

            ;; DECIC
            ((and (eqv? interm-char #\') (eqv? code #x7d))
             (when (cursor-in-scroll-region? state)
               (let ([count (csi-arg-count args 0)]
                     [start-row (scroll-region-top state)]
                     [end-row (scroll-region-bottom state)]
                     [start-col (vterm-state-cursor-col state)]
                     [end-col (scroll-region-right state)])
                  (send-scroll-event state start-row end-row start-col end-col 0 (fx- count)))))

            ;; DECDC
            ((and (eqv? interm-char #\') (eqv? code #x7e))
             (when (cursor-in-scroll-region? state)
               (let ([count (csi-arg-count args 0)]
                     [start-row (scroll-region-top state)]
                     [end-row (scroll-region-bottom state)]
                     [start-col (vterm-state-cursor-col state)]
                     [end-col (scroll-region-right state)])
                  (send-scroll-event state start-row end-row start-col end-col 0 count))))

            (else
              (send-unhandled-seq-event state (list 'csi args interm)))
          )

          ;; bounds check
          (let ([col (vterm-state-cursor-col state)]
                [row (vterm-state-cursor-row state)])
            (lbound row 0)
            (ubound row (- (vterm-state-rows state) 1))
            (vterm-state-cursor-row-set! state row)
 
            (lbound col 0)
            (ubound col (- (vterm-state-current-row-width state) 1))
            (vterm-state-cursor-col-set! state col))

          (update-cursor state old-pos cancel-phantom?)
        )))))

(define make-vterm-state
  (case-lambda
    [()
     (make-vterm-state #f)]

    [(handler)
     (make-vterm-state 25 80 handler) ]

    [(rows cols)
     (make-vterm-state rows cols #f) ]

    [(rows cols handler)
     (let ([state (make-$vterm-state rows cols handler)])
       (vterm-state-parser-set! state
         (make-vterm-parser
           (lambda (seq)
             (cond
               ((string? seq) (handle-text state seq))
               ((list? seq)
                (case (car seq)
                  ((ctl) (handle-ctl state (cadr seq)))
                  ((csi) (handle-csi state (list-ref seq 1)
                                           (list-ref seq 2)
                                           (list-ref seq 3)))
                  (else
                    (send-unhandled-seq-event state seq))
                ))))))
       state)]))

(define (vterm-state-decoder-set! state dec)
  (vterm-parser-decoder-set! (vterm-state-parser state) dec))

(define (vterm-state-utf8-enabled? state)
  (vterm-parser-utf8-enabled? (vterm-state-parser state)))

(define (vterm-state-utf8-enable! state enabled?)
  (vterm-parser-utf8-enable! (vterm-state-parser state) enabled?)
  (if enabled?
      (vterm-state-decoder-set! state (make-utf8-decoder))
    (vterm-state-decoder-set! state (make-ascii-decoder))))

(define vterm-state-reset
  (case-lambda
    [(state)
     (vterm-state-reset state #f)]

    [(state hard?)
     (decode-init (vterm-parser-decoder (vterm-state-parser state)))
     (vterm-state-rows-set! state 25)
     (vterm-state-cols-set! state 80)
     (vterm-state-cursor-set! state (make-vterm-pos 0 0))
     (vterm-state-at-phantom?-set! state #f)
     (vterm-state-scroll-top-set! state 0)
     (vterm-state-scroll-bottom-set! state -1)
     (vterm-state-scroll-left-set! state 0)
     (vterm-state-scroll-right-set! state -1)
     (vterm-state-tab-stops-set! state (make-tab-stops 80))
     (vterm-state-line-info-set! state (make-vector 25 (line-info)))
     (vterm-state-mode-set! state '(auto-wrap))
     (vterm-state-gl-set! state 0)
     (vterm-state-gr-set! state 1)
     (vterm-state-gsingle-set! state 0)
     (vterm-state-default-fg-set! state Vterm-Gray)
     (vterm-state-default-bg-set! state Vterm-Black)
     (vterm-state-colors-set! state ansi-colors)
     (vterm-state-pen-set! state (make-vterm-style))
     (when hard?
       (send-erase-event state 0 (vterm-state-rows state)
                               0 (vterm-state-cols state)
                               #f)) ]))

(define (vterm-state-color-ref state n)
  (vector-ref (vterm-state-colors state) n))

(define (vterm-state-color-set! state n color)
  (vector-set! (vterm-state-colors state) n color))

(define (send-style-event state style)
  (let ([evt (vterm-cached-event-style (vterm-state-cached-event state))])
    (vterm-style-event-fg-set! evt (vterm-style-fg style))
    (vterm-style-event-bg-set! evt (vterm-style-bg style))
    (vterm-style-event-attr-set! evt (vterm-style-attr style))
  (send-state-event state evt)))

(define (vterm-state-pen-fg state)
  (vterm-style-fg (vterm-state-pen state)))

(define (vterm-state-pen-fg-set! state fg)
  (vterm-style-fg-set! (vterm-state-pen state) fg)
  (send-style-event state (vterm-state-pen state)))

(define (vterm-state-pen-bg state)
  (vterm-style-bg (vterm-state-pen state)))

(define (vterm-state-pen-bg-set! state bg)
  (vterm-style-bg-set! (vterm-state-pen state) bg)
  (send-style-event state (vterm-state-pen state)))

(define (vterm-state-pen-attr-on! state attr)
  (vterm-style-attr-on! (vterm-state-pen state) attr)
  (send-style-event state (vterm-state-pen state)))

(define (vterm-state-pen-attr-off! state attr)
  (vterm-style-attr-off! (vterm-state-pen state) attr)
  (send-style-event state (vterm-state-pen state)))

(define (vterm-state-pen-attr-set? state attr)
  (vterm-style-attr-set? (vterm-state-pen state) attr))

(define (vterm-state-pen-attrs-set? state attrs)
  (vterm-style-attrs-set? (vterm-state-pen state) attrs))

(define (vterm-state-pen-reset! state)
  (vterm-style-fg-set! (vterm-state-pen state) (vterm-state-default-fg state))
  (vterm-style-bg-set! (vterm-state-pen state) (vterm-state-default-bg state))
  (vterm-style-attr-reset! (vterm-state-pen state))
  (send-style-event state (vterm-state-pen state)))

(define (vterm-state-process state seq)
  (vterm-parse (vterm-state-parser state) seq))

(define (vterm-state-resize state rows cols)
  (define old-pos (make-vterm-pos (vterm-state-cursor-row state)
                                  (vterm-state-cursor-col state)))

  (when (not (fx=? cols (vterm-state-cols state)))
    (let* ([new-tab-stops (make-tab-stops cols)]
           [old-tab-stops (vterm-state-tab-stops state)])
      (do ([col 0 (fx+ col 1)])
          ((or (fx=? col (vterm-state-cols state))
               (fx=? col cols)))
        (if (is-col-tab-stop? old-tab-stops col)
            (set-col-tab-stop new-tab-stops col)
            (clear-col-tab-stop new-tab-stops col)))
      (vterm-state-tab-stops-set! state new-tab-stops)
  ))

  (when (not (fx=? rows (vterm-state-rows state)))
    (let ([new-line-info (make-vector rows (line-info))]
          [old-line-info (vterm-state-line-info state)])
      (do ([row 0 (fx+ row 1)])
          ((or (fx=? row rows)
              (fx=? row (vterm-state-rows state))))
        (vector-set! new-line-info row (vector-ref old-line-info row)))
      (vterm-state-line-info-set! state new-line-info)
  ))

  (vterm-state-rows-set! state rows)
  (vterm-state-cols-set! state cols)

  (let ([evt (make-vterm-resize-event rows cols (make-vterm-pos 0 0))])
    (send-state-event state evt)
    (when (vterm-state-at-phantom? state)
      (vterm-state-at-phantom?-set! state #f)
      (vterm-state-cursor-col-inc! state 1))

    (vterm-state-cursor-row-inc! state (vterm-pos-row (vterm-resize-event-delta evt)))
    (vterm-state-cursor-col-inc! state (vterm-pos-col (vterm-resize-event-delta evt)))

    (when (fx>=? (vterm-state-cursor-row state) rows)
      (vterm-state-cursor-row-set! state (fx- rows 1)))
    (when (fx>=? (vterm-state-cursor-col state) cols)
      (vterm-state-cursor-col-set! state (fx- cols 1)))

    (update-cursor state old-pos #t)
))

)
