#!/usr/bin/env scheme-script
;; -*- mode: scheme; coding: utf-8 -*- !#
;; Copyright (c) 2023 
;; SPDX-License-Identifier: MIT
#!r6rs

(import (rnrs (6))
        (srfi :64 testing)
        (vterm)
	(vterm common)
	(vterm events)
        (vterm parser)
        (vterm style)
        (vterm state)
        (vterm screen)
        (vterm encoding)
	(text-mode console model)
	(text-mode console events))

(define latin-1-transcoder (make-transcoder (latin-1-codec) 'none 'ignore))

(define (string->ascii str)
  (string->bytevector str latin-1-transcoder))

(define (display-cursor st)
  (display "curs:")(display (vterm-state-cursor-row st))
                   (display ", ")
                   (display (vterm-state-cursor-col st))
		   (newline))

(define state-events '())

(define ($make-test-vterm-state evts)
  (let ([st (make-vterm-state (lambda (st evt)
			         (when (and (vterm-cursor-event? evt) (member 'cursor evts))
				   (let* ([ev (list 'cursor
                                                (vterm-pos-row (vterm-cursor-event-new-pos evt))
                                                (vterm-pos-col (vterm-cursor-event-new-pos evt))
                                                )])
			          (set! state-events (append state-events (list ev)))))
			         (when (and (vterm-scroll-event? evt) (member 'scroll evts))
				   (let* ([rect (vterm-scroll-event-rect evt)]
                                          [ev (list 'scroll
                                                (vterm-rect-start-row rect)
                                                (vterm-rect-end-row rect)
                                                (vterm-rect-start-col rect)
                                                (vterm-rect-end-col rect)
                                                (vterm-scroll-event-downward evt)
                                                (vterm-scroll-event-rightward evt))])
			          (set! state-events (append state-events (list ev)))))
			        (when (and (vterm-erase-event? evt) (member 'erase evts))
			 	  (let* ([rect (vterm-erase-event-rect evt)]
                                         [ev (list 'erase
                                               (vterm-rect-start-row rect)
                                               (vterm-rect-end-row rect)
                                               (vterm-rect-start-col rect)
                                               (vterm-rect-end-col rect))])
                                    (when (vterm-erase-event-selective? evt)
                                      (set! ev (append ev '(selective)))) 
			            (set! state-events (append state-events (list ev)))))
			       (when (and (vterm-glyph-event? evt) (member 'glyph evts))
				 (let ([ev (list 'glyph
                                             (char->integer (vterm-glyph-event-char evt))
                                             (vterm-glyph-event-width evt)
                                             (vterm-pos-row (vterm-glyph-event-pos evt))
                                             (vterm-pos-col (vterm-glyph-event-pos evt))
                                           )])
			         (set! state-events (append state-events (list ev)))))
			       (when (and (vterm-property-event? evt) (member 'property evts))
				 (let ([ev (list 'property
                                             (vterm-property-event-symbol evt)
                                             (vterm-property-event-value evt)
                                           )])
			         (set! state-events (append state-events (list ev)))))
			        ))])
    st))

(define make-test-vterm-state
  (case-lambda
    [(evts)
     (reset-events)
     ($make-test-vterm-state evts)]))

(define screen-events '())

(define (make-test-vterm-screen st evts)
  (let ([scr (make-vterm-screen st
                               (lambda (scr evt)
			         (when (and (vterm-push-line-event? evt) (member 'push-line evts))
				   (set! screen-events
                                         (append screen-events
                                                 (list (list 'push-line
                                                       (filter (lambda (c) (char? c))
                                                         (vector->list
                                                           (vector-map (lambda (c)
                                                                         (vterm-cell-char c))
                                                             (vterm-push-line-event-cells evt)))))))))
			         (when (and (vterm-pop-line-event? evt) (member 'pop-line evts))
                                   (vterm-pop-line-event-has-more?-set! evt #t)
				   (let ([cells (vterm-pop-line-event-cells evt)])
                                     (do ([col 0 (fx+ col 1)]) ((fx=? col (vector-length cells)))
                                       (let ([c (vector-ref cells col)])
                                         (if (fx<? col 5)
                                             (vterm-cell-char-set! c (integer->char (fx+ (char->integer #\A) col)))
                                           (vterm-cell-char-set! c 'null))
                                         (vterm-cell-width-set! c 1)))
				     (set! screen-events (append screen-events (list (list 'pop-line (vector-length cells)))))))
			         (when (and (vterm-cursor-event? evt) (member 'cursor evts))
				   (let* ([ev (list 'cursor
                                                (vterm-pos-row (vterm-cursor-event-new-pos evt))
                                                (vterm-pos-col (vterm-cursor-event-new-pos evt))
                                                )])
			          (set! state-events (append state-events (list ev)))))
			         (when (and (vterm-damage-event? evt) (member 'damage evts))
				   (let* ([rect (vterm-damage-event-rect evt)]
                                          [ev (list 'damage
                                                (vterm-rect-start-row rect)
                                                (vterm-rect-end-row rect)
                                                (vterm-rect-start-col rect)
                                                (vterm-rect-end-col rect)
                                                )])
			          (set! screen-events (append screen-events (list ev))))))
			        )])
  scr))

(define (reset-events)
  (set! screen-events '())
  (set! state-events '()))

(define (reset-state st)
  (vterm-state-reset st 'hard)
  (reset-events))

(test-begin "vterm-seq->sexp")
(let ([p (make-vterm-parser)])
  (test-assert p)

  ;; control sequences
  ;; C0
  (test-equal '((ctl #x03)) (vterm-seq->sexp p #vu8(#x03)))
  (test-equal '((ctl #x1f)) (vterm-seq->sexp p #vu8(#x1f)))
  (test-equal '((ctl #x9f)) (vterm-seq->sexp p #vu8(#x9f)))
  (test-equal '((ctl #x07)) (vterm-seq->sexp p #vu8(#x07)))
  ;; C1 8bit
  (test-equal '((ctl #x83)) (vterm-seq->sexp p #vu8(#x83)))
  (test-equal '((ctl #x9f)) (vterm-seq->sexp p #vu8(#x9f)))
  ;; C1 7bit
  (test-equal '((ctl #x83)) (vterm-seq->sexp p #vu8(#x1b #x43)))
  (test-equal '((ctl #x9f)) (vterm-seq->sexp p #vu8(#x1b #x5f)))

  ;; Escape
  (test-equal '((esc "=")) (vterm-seq->sexp p "\x1b;="))
  ;; Escape 2-byte
  (test-equal '((esc "(X")) (vterm-seq->sexp p "\x1b;(X"))
  ;; Escape split write
  (test-assert (null? (vterm-seq->sexp p "\x1b;(")))
  (test-equal '((esc "(Y")) (vterm-seq->sexp p "Y"))
  ;; Escape cancels Escape, starts another
  (test-equal '((esc ")Z")) (vterm-seq->sexp p "\x1b;(\x1b;)Z"))
  ;; CAN cancels Escape, returns to normal mode
  (test-equal '("AB") (vterm-seq->sexp p "\x1b;(\x18;AB"))
  ;; C0 in Escape interrupts and continues
  (test-assert '((ctl 10) (esc "(X")) (vterm-seq->sexp p "\x1b;(\nX"))

  ;; Text
  (test-equal '("hello") (vterm-seq->sexp p "hello"))
  ;; Mixed sequences
  (test-equal `("1" (ctl ,(char->integer #\newline)) "2") (vterm-seq->sexp p "1\n2"))

  ;; CSI 0 args
  (test-equal '((csi #\a (#f) #f)) (vterm-seq->sexp p "\x1b;[a"))
  ;; CSI 1 arg
  (test-equal '((csi #\b (9) #f)) (vterm-seq->sexp p "\x1b;[9b"))
  ;; CSI 2 args
  (test-equal '((csi #\c (3 4) #f)) (vterm-seq->sexp p "\x1b;[3;4c"))
  ;; CSI 1 arg 1 sub
  (test-equal '((csi #\c (1 (2)) #f)) (vterm-seq->sexp p "\x1b;[1:2c"))
  ;; CSI many digits
  (test-equal '((csi #\d (678) #f)) (vterm-seq->sexp p "\x1b;[678d"))
  ;; CSI leading zero
  (test-equal '((csi #\e (7) #f)) (vterm-seq->sexp p "\x1b;[007e"))
  ;; CSI qmark
  (test-equal '((csi #\f ("?" 2 7) #f)) (vterm-seq->sexp p "\x1b;[?2;7f"))
  ;; CSI greater
  (test-equal '((csi #\c (">" #f) #f)) (vterm-seq->sexp p "\x1b;[>c"))
  ;; CSI SP
  (test-equal '((csi #\q (12) " ")) (vterm-seq->sexp p "\x1b;[12 q"))
  ;; Mixed CSI
  (test-equal '("A" (csi #\m (8) #f) "B") (vterm-seq->sexp p "A\x1b;[8mB"))
  ;; Split write
  (test-assert (null? (vterm-seq->sexp p "\x1b;")))
  (test-equal '((csi #\a (#f) #f)) (vterm-seq->sexp p "[a"))
  (test-equal '("foo") (vterm-seq->sexp p "foo\x1b;["))
  (test-equal '((csi #\b (4) #f)) (vterm-seq->sexp p "4b"))
  (test-assert (null? (vterm-seq->sexp p "\x1b;[12;")))
  (test-equal '((csi #\c (12 3) #f)) (vterm-seq->sexp p "3c"))
  ;; Escape cancels CSI, starts Escape
  (test-equal '((esc "9")) (vterm-seq->sexp p "\x1b;[123\x1b;9"))
  ;; CAN cancels CSI, returns to normal mode
  (test-equal '("AB") (vterm-seq->sexp p "\x1b;[12\x18;AB"))
  ;; C0 in Escape interrupts and continues
  (test-equal '((ctl 10) (csi #\X (12 3) #f)) (vterm-seq->sexp p "\x1b;[12\n;3X"))
  ;; OSC, DCS
  ;; OSC BEL
  (test-equal '((osc "1;Hello")) (vterm-seq->sexp p "\x1b;]1;Hello\x07;"))
  ;; OSC ST (7bit)
  (test-equal '((osc "1;Hello")) (vterm-seq->sexp p "\x1b;]1;Hello\x1b;\\"))
  ;; OSC ST (8bit)
  (test-equal '((osc "1;Hello")) (vterm-seq->sexp p "\x9d;1;Hello\x9c;"))
  ;; Escape cancels OSC, starts Escape
  (test-equal '((esc "9")) (vterm-seq->sexp p "\x1b;]Something\x1b;9"))
  ;; CAN cancels OSC, returns to normal mode
  (test-equal '("AB") (vterm-seq->sexp p "\x1b;]12\x18;AB"))
  ;; C0 in OSC interrupts and continues
  (test-equal '((ctl 10) (osc "2;Bye")) (vterm-seq->sexp p "\x1b;]2;\nBye\x07;"))
  ;; DCS BEL
  (test-equal '((dcs "Hello")) (vterm-seq->sexp p "\x1b;PHello\x07;"))
  ;; DCS ST (7bit)
  (test-equal '((dcs "Hello")) (vterm-seq->sexp p "\x1b;PHello\x1b;\\"))
  ;; DCS ST (8bit)
  (test-equal '((dcs "Hello")) (vterm-seq->sexp p "\x90;Hello\x9c;"))
  ;; Escape cancels DCS, starts Escape
  (test-equal '((esc "9")) (vterm-seq->sexp p "\x1b;PSomething\x1b;9"))
  ;; CAN cancels DCS, returns to normal mode
  (test-equal '("AB") (vterm-seq->sexp p "\x1b;P12\x18;AB"))
  ;; C0 in OSC interrupts and continues
  (test-equal '((ctl 10) (dcs "Bye")) (vterm-seq->sexp p "\x1b;PBy\ne\x07;"))
  ;; NUL ignored
  (test-assert (null? (vterm-seq->sexp p "\x00;")))

  ;; NUL ignored within CSI
  (test-equal '((csi #\m (123) #f)) (vterm-seq->sexp p "\x1b;[12\x00;3m"))
  ;; DEL ignored
  (test-assert (null? (vterm-seq->sexp p "\x7f;")))

  ;; DEL ignored within CSI
  (test-equal '((csi #\m (123) #f)) (vterm-seq->sexp p "\x1b;[12\x7f;3m"))
  ;; DEL inside text
  (test-equal '("AB" "C") (vterm-seq->sexp p "AB\x7f;C"))
)
(test-end)

(test-begin "sexp->vterm-seq")
(let ([p (make-vterm-parser)])
  (test-equal '((ctl 10)) (vterm-seq->sexp p (sexp->vterm-seq '((ctl 10))) ))
  (test-equal '((ctl #x83)) (vterm-seq->sexp p (sexp->vterm-seq '((ctl #x83))) ))
  (test-equal '((ctl 10) (ctl 10)) (vterm-seq->sexp p (sexp->vterm-seq '((ctl 10) (ctl 10))) ))
  (test-equal '((ctl 10) (ctl 12)) (vterm-seq->sexp p (sexp->vterm-seq '((ctl 10) (ctl 12))) ))
  (test-equal '((esc "=")) (vterm-seq->sexp p (sexp->vterm-seq '((esc "="))) ))
  (test-equal '((ctl 10) (esc "=")) (vterm-seq->sexp p (sexp->vterm-seq '((ctl 10) (esc "="))) ))
  (test-equal '((ctl 10) (esc "=") (ctl #x83)) (vterm-seq->sexp p (sexp->vterm-seq '((ctl 10) (esc "=") (ctl #x83))) ))
  (test-equal '("hello") (vterm-seq->sexp p (sexp->vterm-seq '("hello")) ))
  (test-equal '("1" (ctl 10) "2") (vterm-seq->sexp p (sexp->vterm-seq '("1" (ctl 10) "2")) ))
  (test-equal '((csi #\m (1 2) #f)) (vterm-seq->sexp p (sexp->vterm-seq '((csi #\m (1 2)) ))))
  (test-equal '((csi #\m (1 (2)) #f)) (vterm-seq->sexp p (sexp->vterm-seq '((csi #\m (1 (2))) ))))
  (test-equal '((csi #\c (#f) #f)) (vterm-seq->sexp p (sexp->vterm-seq '((csi #\c))) ))
  (test-equal '((csi #\u (1) #f)) (vterm-seq->sexp p (sexp->vterm-seq '((csi #\u (1)))) ))
  (test-equal '((csi #\u (1) "+")) (vterm-seq->sexp p (sexp->vterm-seq '((csi #\u (1) "+"))) ))
  (test-equal '((osc "hello")) (vterm-seq->sexp p (sexp->vterm-seq '((osc "hello")) ) ))
  (test-equal '((dcs "set")) (vterm-seq->sexp p (sexp->vterm-seq '((dcs "set")) ) ))
)
(test-end)

(test-begin "state-cursor basic")
(let* ([st (make-vterm-state)])
  ;; implicit
  (vterm-state-process st "ABC")
  (test-equal 3 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))

  ;; backspace
  (vterm-state-process st (make-string 1 #\backspace))
  (test-equal 2 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))

  ;; horizontal tab
  (vterm-state-process st "\t")
  (test-equal 8 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))

  ;; carriage return
  (vterm-state-process st "\r")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))

  ;; linefeed
  (vterm-state-process st "\n")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 1 (vterm-state-cursor-row st))

  ;; backspace bounded by lefthand edge
  (vterm-state-process st "\x1b;[4;2H")
  (test-equal 1 (vterm-state-cursor-col st))
  (test-equal 3 (vterm-state-cursor-row st))
  (vterm-state-process st (make-string 1 #\backspace))
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 3 (vterm-state-cursor-row st))
  (vterm-state-process st (make-string 1 #\backspace))
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 3 (vterm-state-cursor-row st))

  ;; backspace cancels phantom
  (vterm-state-process st "\x1b;[4;80H")
  (test-equal 79 (vterm-state-cursor-col st))
  (test-equal 3 (vterm-state-cursor-row st))
  (vterm-state-process st "X")
  (test-equal 79 (vterm-state-cursor-col st))
  (test-equal 3 (vterm-state-cursor-row st))
  (vterm-state-process st (make-string 1 #\backspace))
  (test-equal 78 (vterm-state-cursor-col st))
  (test-equal 3 (vterm-state-cursor-row st))

  ;; HT bounded by righthand edge
  (vterm-state-process st "\x1b;[1;78H")
  (test-equal 77 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  (vterm-state-process st "\t")
  (test-equal 79 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  (vterm-state-process st "\t")
  (test-equal 79 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
)
(test-end)

(test-begin "state-cursor index, reverse-index")
(let* ([st (make-vterm-state)])
  ;; index
  (vterm-state-process st "ABC\x1b;D")
  (test-equal 3 (vterm-state-cursor-col st))
  (test-equal 1 (vterm-state-cursor-row st))
  ;; reverse-index
  (vterm-state-process st "\x1b;M")
  (test-equal 3 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  ;; new line
  (vterm-state-process st "\x1b;E")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 1 (vterm-state-cursor-row st))
)
(test-end)

(test-begin "state-cursor movement")
(let* ([st (make-vterm-state)])
  (vterm-state-utf8-enable! st #t)

  ;; cursor down
  (vterm-state-process st "\x1b;[B")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 1 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[3B")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 4 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[0B")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 5 (vterm-state-cursor-row st))
  ;; cursor forward
  (vterm-state-process st "\x1b;[C")
  (test-equal 1 (vterm-state-cursor-col st))
  (test-equal 5 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[3C")
  (test-equal 4 (vterm-state-cursor-col st))
  (test-equal 5 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[0C")
  (test-equal 5 (vterm-state-cursor-col st))
  (test-equal 5 (vterm-state-cursor-row st))
  ;; cursor up
  (vterm-state-process st "\x1b;[A")
  (test-equal 5 (vterm-state-cursor-col st))
  (test-equal 4 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[3A")
  (test-equal 5 (vterm-state-cursor-col st))
  (test-equal 1 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[0A")
  (test-equal 5 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  ;; cursor backward
  (vterm-state-process st "\x1b;[D")
  (test-equal 4 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[3D")
  (test-equal 1 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[0D")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  ;; cursor next line
  (vterm-state-process st "   ")
  (test-equal 3 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[E")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 1 (vterm-state-cursor-row st))
  (vterm-state-process st "   ")
  (test-equal 3 (vterm-state-cursor-col st))
  (test-equal 1 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[2E")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 3 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[0E")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 4 (vterm-state-cursor-row st))
  ;; cursor previous line
  (vterm-state-process st "   ")
  (test-equal 3 (vterm-state-cursor-col st))
  (test-equal 4 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[F")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 3 (vterm-state-cursor-row st))
  (vterm-state-process st "   ")
  (test-equal 3 (vterm-state-cursor-col st))
  (test-equal 3 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[2F")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 1 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[0F")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  ;; cursor horizontal absolute
  (vterm-state-process st "\n")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 1 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[20G")
  (test-equal 19 (vterm-state-cursor-col st))
  (test-equal 1 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[G")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 1 (vterm-state-cursor-row st))
  ;; cursor position
  (vterm-state-process st "\x1b;[10;5H")
  (test-equal 4 (vterm-state-cursor-col st))
  (test-equal 9 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[8H")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 7 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[H")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  ;; cursor position cancels phantom
  (vterm-state-process st "\x1b;[10;78H")
  (test-equal 77 (vterm-state-cursor-col st))
  (test-equal 9 (vterm-state-cursor-row st))
)
(test-end)

(test-begin "cursor bounds checking")
(let* ([st (make-vterm-state)])
  (vterm-state-process st "\x1b;[A")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[D")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[25;80H")
  (test-equal 79 (vterm-state-cursor-col st))
  (test-equal 24 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[B")
  (test-equal 79 (vterm-state-cursor-col st))
  (test-equal 24 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[C")
  (test-equal 79 (vterm-state-cursor-col st))
  (test-equal 24 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[E")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 24 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[H")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[F")
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[999G")
  (test-equal 79 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[99;99H")
  (test-equal 79 (vterm-state-cursor-col st))
  (test-equal 24 (vterm-state-cursor-row st))
)
(test-end)

(test-begin "horizontal/vertical positioning")
(let* ([st (make-vterm-state)])
  ;; horizontal position absolute
  (vterm-state-process st "\x1b;[5`")
  (test-equal 4 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  ;; horizontal position relative
  (vterm-state-process st "\x1b;[3a")
  (test-equal 7 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  ;; horizontal position backward
  (vterm-state-process st "\x1b;[3j")
  (test-equal 4 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  ;; horizontal and vertical position
  (vterm-state-process st "\x1b;[3;3f")
  (test-equal 2 (vterm-state-cursor-col st))
  (test-equal 2 (vterm-state-cursor-row st))
  ;; vertical position absolute
  (vterm-state-process st "\x1b;[5d")
  (test-equal 2 (vterm-state-cursor-col st))
  (test-equal 4 (vterm-state-cursor-row st))
  ;; vertical position relative
  (vterm-state-process st "\x1b;[2e")
  (test-equal 2 (vterm-state-cursor-col st))
  (test-equal 6 (vterm-state-cursor-row st))
  ;; vertical position backward
  (vterm-state-process st "\x1b;[2k")
  (test-equal 2 (vterm-state-cursor-col st))
  (test-equal 4 (vterm-state-cursor-row st))
)
(test-end)

(test-begin "tabs movement")
(let* ([st (make-vterm-state)])
  ;; horizontal tab
  (vterm-state-process st "\t")
  (test-equal 8 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  (vterm-state-process st "   ")
  (test-equal 11 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  (vterm-state-process st "\t")
  (test-equal 16 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  (vterm-state-process st "       ")
  (test-equal 23 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  (vterm-state-process st "\t")
  (test-equal 24 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  (vterm-state-process st "        ")
  (test-equal 32 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  (vterm-state-process st "\t")
  (test-equal 40 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  ;; cursor horizontal tab
  (vterm-state-process st "\x1b;[I")
  (test-equal 48 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[2I")
  (test-equal 64 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  ;; cursor backward tab
  (vterm-state-process st "\x1b;[Z")
  (test-equal 56 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
  (vterm-state-process st "\x1b;[2Z")
  (test-equal 40 (vterm-state-cursor-col st))
  (test-equal 0 (vterm-state-cursor-row st))
)
(test-end)

(test-begin "encoding ascii")
(let ([dec (make-ascii-decoder)])
  (let-values ([(cpl bcount) (decode-text dec #vu8(#x31 #x32))])
    (test-equal '(#x31 #x32) cpl)
    (test-equal 2 bcount) ))
(test-end)

(test-begin "encoding utf8")
(let ([dec (make-utf8-decoder)])
  ;; Low
  (let-values ([(cpl _) (decode-text dec (string->utf8 "123"))])
    (test-equal '(#x31 #x32 #x33) cpl))
  ;; 2 byte
  (let-values ([(cpl _) (decode-text dec #vu8(#xC2 #x80 #xDF #xBF))])
    (test-equal '(#x0080 #x07FF) cpl))
  ;; 3 byte
  (let-values ([(cpl _) (decode-text dec #vu8(#xE0 #xA0 #x80 #xEF #xBF #xBD))])
    (test-equal '(#x0800 #xFFFD) cpl))
  ;; 4 byte
  (let-values ([(cpl _) (decode-text dec #vu8(#xF0 #x90 #x80 #x80 #xF7 #xBF #xBF #xBF))])
    (test-equal '(#x10000 #x1FFFFF) cpl))
  ;; Early termination
  (let-values ([(cpl _) (decode-text dec (string->ascii "\xC2;!"))])
    (test-equal '(#xFFFD #x21) cpl))
  (let-values ([(cpl _) (decode-text dec (string->ascii "\xE0;!\xE0;\xA0;!"))])
    (test-equal '(#xFFFD #x21 #xFFFD #x21) cpl))
  (let-values ([(cpl _) (decode-text dec (string->ascii "\xF0;!\xF0;\x90;!\xF0;\x90;\x80;!"))])
    (test-equal '(#xFFFD #x21 #xFFFD #x21 #xFFFD #x21) cpl))
  ;; Early restart
  (let-values ([(cpl _) (decode-text dec (string->ascii "\xC2;\xC2;\x90;"))])
    (test-equal '(#xFFFD #x0090) cpl))
  (let-values ([(cpl _) (decode-text dec (string->ascii "\xE0;\xC2;\x90;\xE0;\xA0;\xC2;\x90;"))])
    (test-equal '(#xFFFD #x0090 #xFFFD #x0090) cpl))
  (let-values ([(cpl _) (decode-text dec (string->ascii "\xE0;\xC2;\x90;\xF0;\x90;\xC2;\x90;\xF0;\x90;\x80;\xC2;\x90;"))])
    (test-equal '(#xFFFD #x0090 #xFFFD #x0090 #xFFFD #x0090) cpl))
  ;; Overlong
  (let-values ([(cpl _) (decode-text dec (string->ascii "\xC0;\x80;\xC1;\xBF;"))])
    (test-equal '(#xFFFD #xFFFD) cpl))
  (let-values ([(cpl _) (decode-text dec (string->ascii "\xE0;\x80;\x80;\xE0;\x9F;\xBF;"))])
    (test-equal '(#xFFFD #xFFFD) cpl))
  (let-values ([(cpl _) (decode-text dec (string->ascii "\xF0;\x80;\x80;\x80;\xF0;\x8F;\xBF;\xBF;"))])
    (test-equal '(#xFFFD #xFFFD) cpl))
  ;; UTF-16 surrogates
  (let-values ([(cpl _) (decode-text dec (string->ascii "\xED;\xA0;\x80;\xED;\xBF;\xBF;"))])
    (test-equal '(#xFFFD #xFFFD) cpl))
  ;; Split write
  (begin
    (let-values ([(cpl _) (decode-text dec (string->ascii "\xC2;"))])
      (test-equal '() cpl))
    (let-values ([(cpl _) (decode-text dec (string->ascii "\xA0;"))])
      (test-equal '(#x00A0) cpl)))
  (begin
    (let-values ([(cpl _) (decode-text dec (string->ascii "\xE0;"))])
      (test-equal '() cpl))
    (let-values ([(cpl _) (decode-text dec (string->ascii "\xA0;\x80;"))])
      (test-equal '(#x00800) cpl)))
  (begin
    (let-values ([(cpl bc) (decode-text dec (string->ascii "\xE0;\xA0;"))])
      (test-equal '() cpl))
    (let-values ([(cpl _) (decode-text dec (string->ascii "\x80;"))])
      (test-equal '(#x00800) cpl)))
  (begin
    (let-values ([(cpl _) (decode-text dec (string->ascii "\xF0;"))])
      (test-equal '() cpl))
    (let-values ([(cpl _) (decode-text dec (string->ascii "\x90;\x80;\x80;"))])
      (test-equal '(#x10000) cpl)))
  (begin
    (let-values ([(cpl _) (decode-text dec (string->ascii "\xF0;\x90;"))])
      (test-equal '() cpl))
    (let-values ([(cpl _) (decode-text dec (string->ascii "\x80;\x80;"))])
      (test-equal '(#x10000) cpl)))
  (begin
    (let-values ([(cpl _) (decode-text dec (string->ascii "\xF0;\x90;\x80;"))])
      (test-equal '() cpl))
    (let-values ([(cpl _) (decode-text dec (string->ascii "\x80;"))])
      (test-equal '(#x10000) cpl))
    (let-values ([(cpl _) (decode-text dec (string->utf8 "Ї"))])
      (test-equal '(#x407) cpl))
    (let-values ([(cpl _) (decode-text dec (string->utf8 "Ґ"))])
      (test-equal '(#x490) cpl)))
)
(test-end)

(test-begin "put-glyph")
(let* ([st (make-test-vterm-state '(glyph))])
  (vterm-state-utf8-enable! st #t)

  ;; Low
  (vterm-state-process st "ABC")
  (test-equal state-events '((glyph #x41 1 0 0)
  		             (glyph #x42 1 0 1)
                             (glyph #x43 1 0 2)))
  (reset-state st)

  ;; UTF-8 1 char
  (vterm-state-process st #vu8(#xC3 #x81 #xC3 #xA9))
  (test-equal state-events '((glyph #xC1 1 0 0)
		             (glyph #xE9 1 0 1)))
  (reset-state st)

  ;; UTF-8 split writes
  (vterm-state-process st #vu8(#xC3))
  (vterm-state-process st #vu8(#x81))
  (test-equal state-events '((glyph #xC1 1 0 0)))
  (reset-state st)

  ;; UTF-8 wide char
  (vterm-state-process st (string->ascii "\xEF;\xBC;\x90; "))
  (test-equal state-events '((glyph #xFF10 2 0 0)
		             (glyph #x20   1 0 2)))
  (reset-state st)

  ;; UTF-8 emoji wide char
  (vterm-state-process st (string->ascii "\xF0;\x9F;\x98;\x80; "))
  (test-equal state-events '((glyph #x1f600 2 0 0)
                             (glyph #x20     1 0 2)))
  (reset-state st)

  ;; TODO combining chars
)
(test-end)

(test-begin "state pen style")
(let ([st (make-vterm-state)])
  (vterm-state-utf8-enable! st #t)

  ;; Reset
  (vterm-state-process st "\x1b;[m")
  (test-assert (not (vterm-state-pen-attr-set? st Vterm-Bold)))
  (test-assert (not (vterm-state-pen-attr-set? st Vterm-Underline)))
  (test-assert (not (vterm-state-pen-attr-set? st Vterm-Italic)))
  (test-assert (not (vterm-state-pen-attr-set? st Vterm-Blink)))
  (test-assert (not (vterm-state-pen-attr-set? st Vterm-Reverse)))
  (test-equal Vterm-Gray (vterm-state-pen-fg st))
  (test-equal Vterm-Black (vterm-state-pen-bg st))
  ;; Bold
  (vterm-state-process st "\x1b;[1m")
  (test-assert (vterm-state-pen-attr-set? st Vterm-Bold))
  (vterm-state-process st "\x1b;[22m")
  (test-assert (not (vterm-state-pen-attr-set? st Vterm-Bold)))
  (vterm-state-process st "\x1b;[1m\x1b;[m")
  (test-assert (not (vterm-state-pen-attr-set? st Vterm-Bold)))

  ;; TODO Underline

  ;; Italic
  (vterm-state-process st "\x1b;[3m")
  (test-assert (vterm-state-pen-attr-set? st Vterm-Italic))
  (vterm-state-process st "\x1b;[23m")
  (test-assert (not (vterm-state-pen-attr-set? st Vterm-Italic)))
  (vterm-state-process st "\x1b;[3m\x1b;[m")
  (test-assert (not (vterm-state-pen-attr-set? st Vterm-Italic)))

  ;; Blink
  (vterm-state-process st "\x1b;[5m")
  (test-assert (vterm-state-pen-attr-set? st Vterm-Blink))
  (vterm-state-process st "\x1b;[25m")
  (test-assert (not (vterm-state-pen-attr-set? st Vterm-Blink)))
  (vterm-state-process st "\x1b;[5m\x1b;[m")
  (test-assert (not (vterm-state-pen-attr-set? st Vterm-Blink)))

  ;; Foreground
  (vterm-state-process st "\x1b;[31m")
  (test-equal (vterm-state-pen-fg st) (vterm-state-color-ref st 1))
  (vterm-state-process st "\x1b;[32m")
  (test-equal (vterm-state-pen-fg st) (vterm-state-color-ref st 2))
  (vterm-state-process st "\x1b;[34m")
  (test-equal (vterm-state-pen-fg st) (vterm-state-color-ref st 4))
  (vterm-state-process st "\x1b;[39m")
  (test-equal (vterm-state-pen-fg st) (vterm-state-default-fg st))

  ;; Background
  (vterm-state-process st "\x1b;[41m")
  (test-equal (vterm-state-pen-bg st) (vterm-state-color-ref st 1))
  (vterm-state-process st "\x1b;[42m")
  (test-equal (vterm-state-pen-bg st) (vterm-state-color-ref st 2))
  (vterm-state-process st "\x1b;[44m")
  (test-equal (vterm-state-pen-bg st) (vterm-state-color-ref st 4))
  (vterm-state-process st "\x1b;[49m")
  (test-equal (vterm-state-pen-bg st) (vterm-state-default-bg st))
)
(test-end)

(test-begin "state scroll")
(let* ([st (make-test-vterm-state '(scroll))])
  (vterm-state-utf8-enable! st #t)

  ;; Linefeed
  (do ([i 0 (fx+ i 1)]) ((fx=? i 24))
    (vterm-state-process st "\n"))

  (test-equal 24 (vterm-state-cursor-row st))
  (test-equal 0 (vterm-state-cursor-col st))
  (vterm-state-process st "\n")
  (test-equal state-events '((scroll 0 25  0 80  1 0)))
  (reset-state st)

  ;; Index
  (vterm-state-process st "\x1b;[25H")
  (vterm-state-process st "\x1b;D")
  (test-equal state-events '((scroll 0 25  0 80  1 0)))
  (reset-state st)

  ;; Reverse Index
  (vterm-state-process st "\x1b;M")
  (test-equal state-events '((scroll 0 25  0 80  -1 0)))
  (reset-state st)

  ;; Linefeed in DECSTBM
  (vterm-state-process st "\x1b;[1;10r")
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 0 (vterm-state-cursor-col st))
  (vterm-state-process st "\n\n\n\n\n\n\n\n\n")
  (test-equal 9 (vterm-state-cursor-row st))
  (test-equal 0 (vterm-state-cursor-col st))
  (reset-events)
  (vterm-state-process st "\n")
  (test-equal state-events '((scroll 0 10  0 80 1 0)))
  (test-equal 9 (vterm-state-cursor-row st))
  (test-equal 0 (vterm-state-cursor-col st))

  ;; Linefeed outside DECSTBM
  (vterm-state-process st "\x1b;[20H")
  (test-equal 19 (vterm-state-cursor-row st))
  (test-equal 0 (vterm-state-cursor-col st))
  (vterm-state-process st "\n")
  (test-equal 20 (vterm-state-cursor-row st))
  (test-equal 0 (vterm-state-cursor-col st))

  ;; Index in DECSTBM
  (vterm-state-process st "\x1b;[9;10r")
  (vterm-state-process st "\x1b;[10H")
  (vterm-state-process st "\x1b;M")
  (test-equal 8 (vterm-state-cursor-row st))
  (test-equal 0 (vterm-state-cursor-col st))
  (reset-events)
  (vterm-state-process st "\x1b;M")
  (test-equal state-events '((scroll 8 10  0 80 -1 0)))

  ;; Reverse Index in DECSTBM
  (vterm-state-process st "\x1b;[25H")
  (test-equal 24 (vterm-state-cursor-row st))
  (test-equal 0 (vterm-state-cursor-col st))
  (reset-events)
  (vterm-state-process st "\n")
  (test-equal state-events '())
  (test-equal 24 (vterm-state-cursor-row st))
  (test-equal 0 (vterm-state-cursor-col st))

  (reset-state st)

  ;; Scroll Down
  (vterm-state-process st "\x1b;[S")
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal state-events '((scroll 0 25  0 80  1 0)))

  (reset-events)
  (vterm-state-process st "\x1b;[2S")
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal state-events '((scroll 0 25  0 80  2 0)))

  (reset-events)
  (vterm-state-process st "\x1b;[100S")
  (test-equal state-events '((scroll 0 25  0 80  25 0)))

  ;; Scroll Up
  (reset-events)
  (vterm-state-process st "\x1b;[T")
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal state-events '((scroll 0 25  0 80  -1 0)))

  (reset-events)
  (vterm-state-process st "\x1b;[2T")
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 0 (vterm-state-cursor-col st))
  (test-equal state-events '((scroll 0 25  0 80  -2 0)))

  (reset-events)
  (vterm-state-process st "\x1b;[100T")
  (test-equal state-events '((scroll 0 25  0 80  -25 0)))

  ;; Scroll Up/Down in DECSTBM
  (vterm-state-process st "\x1b;[5;20r")
  (reset-events)
  (vterm-state-process st "\x1b;[S")
  (test-equal state-events '((scroll 4 20  0 80  1 0)))
  (reset-events)
  (vterm-state-process st "\x1b;[T")
  (test-equal state-events '((scroll 4 20  0 80  -1 0)))

  (reset-state st)
)
(test-end)

(test-begin "state edit")
(let* ([st (make-test-vterm-state '(scroll erase))])
  (vterm-state-utf8-enable! st #t)

  ;; ICH
  (vterm-state-process st "ACD")
  (vterm-state-process st "\x1b;[2D")
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[@")
  (test-equal state-events '((scroll 0 1  1 80  0 -1)))
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "B")
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 2 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[3@")
  (test-equal state-events '((scroll 0 1  2 80  0 -3)))

  ;; DCH
  (reset-state st)
  (vterm-state-process st "ABBC")
  (vterm-state-process st "\x1b;[3D")
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[P")
  (test-equal state-events '((scroll 0 1  1 80  0 1)))
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[3P")
  (test-equal state-events '((scroll 0 1  1 80  0 3)))
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  ;; ECH
  (reset-state st)
  (vterm-state-process st "ABC")
  (vterm-state-process st "\x1b;[2D")
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[X")
  (test-equal state-events '((erase 0 1  1 2)))
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[3X")
  (test-equal state-events '((erase 0 1  1 4)))
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[100X")
  (test-equal state-events '((erase 0 1  1 80)))
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  ;; IL
  (vterm-state-process st "A\r\nC")
  (test-equal 1 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[L")
  (test-equal state-events '((scroll 1 25  0 80  -1 0)))
  (test-equal 1 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  (vterm-state-process st "\rB")
  (test-equal 1 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[3L")
  (test-equal state-events '((scroll 1 25  0 80  -3 0)))
  (test-equal 1 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  ;; DL
  (reset-state st)
  (vterm-state-process st "A\r\nB\r\nB\r\nC")
  (test-equal 3 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  (vterm-state-process st "\x1b;[2H")
  (test-equal 1 (vterm-state-cursor-row st))
  (test-equal 0 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[M")
  (test-equal state-events '((scroll 1 25  0 80  1 0)))
  (test-equal 1 (vterm-state-cursor-row st))
  (test-equal 0 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[3M")
  (test-equal state-events '((scroll 1 25  0 80  3 0)))
  (test-equal 1 (vterm-state-cursor-row st))
  (test-equal 0 (vterm-state-cursor-col st))

  ;; DECIC
  (reset-state st)
  (vterm-state-process st "\x1b;[20G\x1b;[5'}")
  (test-equal state-events '((scroll 0 25  19 80  0 -5)))

  ;; DECDC
  (reset-state st)
  (vterm-state-process st "\x1b;[20G\x1b;[5'~")
  (test-equal state-events '((scroll 0 25  19 80  0 5)))

  ;; EL 0
  (reset-state st)
  (vterm-state-process st "ABCDE")
  (vterm-state-process st "\x1b;[3D")
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 2 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[0K")
  (test-equal state-events '((erase 0 1  2 80)))
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 2 (vterm-state-cursor-col st))

  ;; EL 1
  (reset-state st)
  (vterm-state-process st "ABCDE")
  (vterm-state-process st "\x1b;[3D")
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 2 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[1K")
  (test-equal state-events '((erase 0 1  0 3)))
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 2 (vterm-state-cursor-col st))

  ;; EL 2
  (reset-state st)
  (vterm-state-process st "ABCDE")
  (vterm-state-process st "\x1b;[3D")
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 2 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[2K")
  (test-equal state-events '((erase 0 1  0 80)))
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 2 (vterm-state-cursor-col st))

  ;; SEL
  (reset-state st)
  (vterm-state-process st "ABCDE")
  (vterm-state-process st "\x1b;[11G")
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 10 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[?0K")
  (test-equal state-events '((erase 0 1  10 80 selective)))
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 10 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[?1K")
  (test-equal state-events '((erase 0 1  0 11 selective)))
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 10 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[?2K")
  (test-equal state-events '((erase 0 1  0 80 selective)))
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 10 (vterm-state-cursor-col st))

  ;; ED 0
  (reset-state st)
  (vterm-state-process st "\x1b;[2;2H")
  (test-equal 1 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[0J")
  (test-equal state-events '((erase 1 2  1 80)
                             (erase 2 25 0 80)))
  (test-equal 1 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  ;; ED 1
  (reset-state st)
  (vterm-state-process st "\x1b;[2;2H")
  (test-equal 1 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[1J")
  (test-equal state-events '((erase 0 1  0 80)
                             (erase 1 2  0 2)))
  (test-equal 1 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  ;; ED 2
  (reset-state st)
  (vterm-state-process st "\x1b;[2;2H")
  (test-equal 1 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[2J")
  (test-equal state-events '((erase 0 25  0 80)))
  (test-equal 1 (vterm-state-cursor-row st))
  (test-equal 1 (vterm-state-cursor-col st))

  ;; SED
  (reset-state st)
  (vterm-state-process st "\x1b;[5;5H")
  (test-equal 4 (vterm-state-cursor-row st))
  (test-equal 4 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[?0J")
  (test-equal state-events '((erase 4 5  4 80 selective)
                             (erase 5 25 0 80 selective)))
  (test-equal 4 (vterm-state-cursor-row st))
  (test-equal 4 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[?1J")
  (test-equal state-events '((erase 0 4  0 80 selective)
                             (erase 4 5  0 5 selective)))
  (test-equal 4 (vterm-state-cursor-row st))
  (test-equal 4 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "\x1b;[?2J")
  (test-equal state-events '((erase 0 25  0 80 selective)))
  (test-equal 4 (vterm-state-cursor-row st))
  (test-equal 4 (vterm-state-cursor-col st))
)
(test-end)

(test-begin "screen ascii")
(let* ([st (make-test-vterm-state '())]
       [scr (make-test-vterm-screen st '(cursor))])
  (vterm-state-process st "ABC")
  (test-equal '(#\A #\B #\C) (vterm-screen-chars scr 0 0 1 3))
  (test-equal '(#\A #\B #\C) (vterm-screen-chars scr 0 0 1 80))
  (test-assert (vterm-cell=? (vterm-screen-ref scr 0 0)
                             (make-vterm-cell #\A)))
  (test-assert (vterm-cell=? (vterm-screen-ref scr 0 1)
                             (make-vterm-cell #\B)))
  (test-assert (vterm-cell=? (vterm-screen-ref scr 0 2)
                             (make-vterm-cell #\C)))

  (reset-events)
  (vterm-state-process st "\x1b;[H")
  (test-equal state-events '((cursor 0 0)))
  (test-equal '(#\A #\B #\C) (vterm-screen-chars scr 0 0 1 3))
  (test-equal '(#\A #\B #\C) (vterm-screen-chars scr 0 0 1 80))

  (reset-events)
  (vterm-state-process st "E")
  (test-equal state-events '((cursor 0 1)))
  (test-equal '(#\E #\B #\C) (vterm-screen-chars scr 0 0 1 80))

  ;; Erase
  (reset-state st)
  (vterm-state-process st "ABCDE\x1b;[H\x1b;[K")
  (test-equal '() (vterm-screen-chars scr 0 0 1 80))

  ;; Copycell
  (reset-state st)
  (vterm-state-process st "ABC\x1b;[H\x1b;[@")
  (vterm-state-process st "1")
  (test-equal '(#\1 #\A #\B #\C) (vterm-screen-chars scr 0 0 1 80))

  (reset-state st)
  (vterm-state-process st "ABC\x1b;[H\x1b;[P")
  (test-equal '(#\B) (vterm-screen-chars scr 0 0 1 1))
  (test-equal '(#\C) (vterm-screen-chars scr 0 1 1 2))
  (test-equal '(#\B #\C) (vterm-screen-chars scr 0 0 1 80))

  ;; Space padding
  (reset-state st)
  (vterm-state-process st "Hello\x1b;[CWorld")
  (test-equal '(#\H #\e #\l #\l #\o #\space #\W #\o #\r #\l #\d) (vterm-screen-chars scr 0 0 1 80))

  ;; Linefeed padding
  (reset-state st)
  (vterm-state-process st "Hello\r\nWorld")
  (test-equal '(#\H #\e #\l #\l #\o #\newline #\W #\o #\r #\l #\d) (vterm-screen-chars scr 0 0 2 80))
  (test-equal "Hello\nWorld" (vterm-screen-string scr 0 0 2 80))
)
(test-end)

(test-begin "screen damage")
(let* ([st (make-test-vterm-state '())]
       [scr (make-test-vterm-screen st '(damage))])

  ;; Putglyph
  (vterm-state-process st "123")
  (test-equal screen-events '((damage 0 1  0 1)
                              (damage 0 1  1 2)
                              (damage 0 1  2 3)))
  ;; Erase
  (reset-events)
  (vterm-state-process st "\x1b;[H")
  (vterm-state-process st "\x1b;[3X")
  (test-equal screen-events '((damage 0 1  0 3)))

  ;; Scroll damages entire line in two chunks
  (reset-events)
  (vterm-state-process st "\x1b;[H\x1b;[5@")
  (test-equal screen-events '((damage 0 1  5 80)
                              (damage 0 1  0 5)))

  ;; Scroll down damages entire screen in two chunks
  (reset-events)
  (vterm-state-process st "\x1b;[T")
  (test-equal screen-events '((damage 1 25  0 80)
                              (damage 0 1   0 80)))
)
(test-end)

(test-begin "terminal property")
(let* ([st (make-test-vterm-state '(property))])
  (vterm-state-process st "\x1b;[?25h")
  (test-equal state-events '((property cursor on)))

  (reset-events)
  (vterm-state-process st "\x1b;[?25l")
  (test-equal state-events '((property cursor off)))
)
(test-end)

(define (->chars bv)
  (map (lambda (v)
	 (integer->char v))
       (bytevector->u8-list bv)))

(test-begin "terminal input")
(let-values ([(op g) (open-bytevector-output-port)])
  (let* ([vt (make-vterm)])
    (vterm-output-port-set! vt op)

    ;; Unmodified ASCII
    (vterm-write-char vt (integer->char #x41))
    (test-equal #vu8(#x41) (g))

    (vterm-write-char vt (integer->char #x61))
    (test-equal #vu8(#x61) (g))

    ;; Shift modifier on upper ASCII letters
    (vterm-write-char vt #\A (modifier-set shift))
    (test-equal '(#\A) (->chars (g)))

    ;; Ctrl modifier on ASCII letters
    (vterm-write-char vt (integer->char #x41) (modifier-set ctrl))
    (test-equal '(#\esc #\[ #\6 #\5 #\; #\5 #\u) (->chars (g)))

    (vterm-write-char vt (integer->char #x61) (modifier-set ctrl))
    (test-equal #vu8(#x1) (g))

    ;; Alt modifier on ASCII letters
    (vterm-write-char vt (integer->char #x41) (modifier-set alt))
    (test-equal '(#\esc #\A) (->chars (g)))

    (vterm-write-char vt (integer->char #x61) (modifier-set alt))
    (test-equal '(#\esc #\a) (->chars (g)))

    ;; Ctrl+Alt modifier on ASCII letter
    (vterm-write-char vt (integer->char #x41) (modifier-set ctrl alt))
    (test-equal '(#\esc #\[ #\6 #\5 #\; #\7 #\u) (->chars (g)))

    (vterm-write-char vt (integer->char #x61) (modifier-set ctrl alt))
    (test-equal #vu8(#x1b #x1) (g))

    ;; Special handling of Ctrl-I
    (vterm-write-char vt (integer->char #x49))
    (test-equal '(#\I) (->chars (g)))

    (vterm-write-char vt (integer->char #x69))
    (test-equal '(#\i) (->chars (g)))

    (vterm-write-char vt (integer->char #x49) (modifier-set ctrl))
    (test-equal '(#\esc #\[ #\7 #\3 #\; #\5 #\u) (->chars (g)))

    (vterm-write-char vt (integer->char #x69) (modifier-set ctrl))
    (test-equal '(#\esc #\[ #\1 #\0 #\5 #\; #\5 #\u) (->chars (g)))

    (vterm-write-char vt (integer->char #x49) (modifier-set alt))
    (test-equal '(#\esc #\I) (->chars (g)))

    (vterm-write-char vt (integer->char #x69) (modifier-set alt))
    (test-equal '(#\esc #\i) (->chars (g)))

    (vterm-write-char vt (integer->char #x49) (modifier-set ctrl alt))
    (test-equal '(#\esc #\[ #\7 #\3 #\; #\7 #\u) (->chars (g)))

    (vterm-write-char vt (integer->char #x69) (modifier-set ctrl alt))
    (test-equal '(#\esc #\[ #\1 #\0 #\5 #\; #\7 #\u) (->chars (g)))

    ;; Special handling of Space
    (vterm-write-char vt (integer->char #x20))
    (test-equal '(#\space) (->chars (g)))

    (vterm-write-char vt (integer->char #x20) (modifier-set shift))
    (test-equal '(#\esc #\[ #\3 #\2 #\; #\2 #\u) (->chars (g)))

    (vterm-write-char vt (integer->char #x20) (modifier-set ctrl))
    (test-equal #vu8(#x0) (g))

    (vterm-write-char vt (integer->char #x20) (modifier-set shift ctrl))
    (test-equal '(#\esc #\[ #\3 #\2 #\; #\6 #\u) (->chars (g)))

    (vterm-write-char vt (integer->char #x20) (modifier-set alt))
    (test-equal '(#\esc #\space) (->chars (g)))

    (vterm-write-char vt (integer->char #x20) (modifier-set shift alt))
    (test-equal '(#\esc #\[ #\3 #\2 #\; #\4 #\u) (->chars (g)))

    (vterm-write-char vt (integer->char #x20) (modifier-set ctrl alt))
    (test-equal #vu8(#x1b #x0) (g))

    (vterm-write-char vt (integer->char #x20) (modifier-set shift ctrl alt))
    (test-equal '(#\esc #\[ #\3 #\2 #\; #\8 #\u) (->chars (g)))

    ;; Cursor keys in reset mode
    (vterm-write-key vt 'ArrowUp)
    (test-equal '(#\esc #\[ #\A) (->chars (g)))

    (vterm-write-key vt 'ArrowUp (modifier-set shift))
    (test-equal '(#\esc #\[ #\1 #\; #\2 #\A) (->chars (g)))

    (vterm-write-key vt 'ArrowUp (modifier-set ctrl))
    (test-equal '(#\esc #\[ #\1 #\; #\5 #\A) (->chars (g)))

    (vterm-write-key vt 'ArrowUp (modifier-set shift ctrl))
    (test-equal '(#\esc #\[ #\1 #\; #\6 #\A) (->chars (g)))

    (vterm-write-key vt 'ArrowUp (modifier-set alt))
    (test-equal '(#\esc #\[ #\1 #\; #\3 #\A) (->chars (g)))

    (vterm-write-key vt 'ArrowUp (modifier-set shift alt))
    (test-equal '(#\esc #\[ #\1 #\; #\4 #\A) (->chars (g)))

    (vterm-write-key vt 'ArrowUp (modifier-set ctrl alt))
    (test-equal '(#\esc #\[ #\1 #\; #\7 #\A) (->chars (g)))

    (vterm-write-key vt 'ArrowUp (modifier-set ctrl alt shift))
    (test-equal '(#\esc #\[ #\1 #\; #\8 #\A) (->chars (g)))

    ;; Cursor keys in application mode
    (vterm-process vt "\x1b;[?1h")

    (vterm-write-key vt 'ArrowUp)
    (test-equal '(#\esc #\O #\A) (->chars (g)))

    (vterm-write-key vt 'ArrowUp (modifier-set shift))
    (test-equal '(#\esc #\[ #\1 #\; #\2 #\A) (->chars (g)))

    (vterm-write-key vt 'ArrowUp (modifier-set ctrl))
    (test-equal '(#\esc #\[ #\1 #\; #\5 #\A) (->chars (g)))

    ;; Shift-Tab should be different
    (vterm-write-key vt 'Tab)
    (test-equal #vu8(#x9) (g))

    (vterm-write-key vt 'Tab (modifier-set shift))
    (test-equal '(#\esc #\[ #\Z) (->chars (g)))

    (vterm-write-key vt 'Tab (modifier-set ctrl))
    (test-equal '(#\esc #\[ #\9 #\; #\5 #\u) (->chars (g)))

    (vterm-write-key vt 'Tab (modifier-set alt))
    (test-equal #vu8(#x1b #x9) (g))

    (vterm-write-key vt 'Tab (modifier-set ctrl alt))
    (test-equal '(#\esc #\[ #\9 #\; #\7 #\u) (->chars (g)))

    ;; Enter in linefeed mode
    (vterm-write-key vt 'Enter)
    (test-equal #vu8(#xd) (g))

    ;; Enter in newline mode
    (vterm-process vt "\x1b;[20h")
    (vterm-write-key vt 'Enter)
    (test-equal #vu8(#xd #xa) (g))

    ;; !Unmodified F1 is SS3 P
    (vterm-write-key vt 'F1)
    (test-equal '(#\esc #\O #\P) (->chars (g)))
))
(test-end)

(test-begin "state resizeh")
(let* ([st (make-test-vterm-state '(glyph))])
  (vterm-state-utf8-enable! st #t)

  (vterm-state-process st "AB\x1b;[79GCDE")
  (test-equal state-events '((glyph #x41 1 0 0)
  		             (glyph #x42 1 0 1)
                             (glyph #x43 1 0 78)
                             (glyph #x44 1 0 79)
                             (glyph #x45 1 1 0)))
  ;; Resize with reset
  (reset-state st)
  (vterm-state-resize st 27 85)

  (vterm-state-process st "AB\x1b;[79GCDE")
  (test-equal state-events '((glyph #x41 1 0 0)
  		             (glyph #x42 1 0 1)
                             (glyph #x43 1 0 78)
                             (glyph #x44 1 0 79)
                             (glyph #x45 1 0 80)))
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 81 (vterm-state-cursor-col st))

  ;; Resize without reset
  (vterm-state-resize st 28 90)

  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 81 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "FGHI")
  (test-equal state-events '((glyph #x46 1 0 81)
  		             (glyph #x47 1 0 82)
                             (glyph #x48 1 0 83)
                             (glyph #x49 1 0 84)))
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 85 (vterm-state-cursor-col st))

  ;; Resize shrink moves cursor
  (vterm-state-resize st 25 80)

  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 79 (vterm-state-cursor-col st))

  ;; Resize grow does not cancel phantom
  (reset-state st)
  (vterm-state-process st "\x1b;[79GAB")
  (test-equal state-events '((glyph #x41 1 0 78)
                             (glyph #x42 1 0 79)))
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 79 (vterm-state-cursor-col st))

  (vterm-state-resize st 30 100)
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 80 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-state-process st "C")
  (test-equal state-events '((glyph #x43 1 0 80)))
  (test-equal 0 (vterm-state-cursor-row st))
  (test-equal 81 (vterm-state-cursor-col st))
)
(test-end)

(test-begin "screen resize")
(let* ([st (make-test-vterm-state '())]
       [scr (make-test-vterm-screen st '(push-line))])
  ;; Resize wider preserves cells
  (vterm-screen-resize scr 25 80)
  (vterm-state-process st "AB\r\nCD")
  (test-equal '(#\A #\B) (vterm-screen-chars scr 0 0 1 80))
  (test-equal '(#\C #\D) (vterm-screen-chars scr 1 0 2 80))

  (vterm-screen-resize scr 25 100)
  (test-equal '(#\A #\B) (vterm-screen-chars scr 0 0 1 100))
  (test-equal '(#\C #\D) (vterm-screen-chars scr 1 0 2 100))

  ;; Resize wider allows print in new area
  (reset-state st)
  (vterm-screen-resize scr 25 80)
  (vterm-state-process st "AB\x1b;[79GCD")
  (test-equal '(#\A #\B) (vterm-screen-chars scr 0 0 1 2))
  (test-equal '(#\C #\D) (vterm-screen-chars scr 0 78 1 80))

  (vterm-screen-resize scr 25 100)
  (test-equal '(#\A #\B) (vterm-screen-chars scr 0 0 1 2))
  (test-equal '(#\C #\D) (vterm-screen-chars scr 0 78 1 80))

  (vterm-state-process st "E")
  (test-equal '(#\C #\D #\E) (vterm-screen-chars scr 0 78 1 81))

  ;; Resize shorter with blanks just truncates
  (reset-state st)
  (vterm-screen-resize scr 25 80)

  (vterm-state-process st "Top\x1b;[10HLine 10")
  (test-equal '(#\T #\o #\p) (vterm-screen-chars scr 0 0 1 80))
  (test-equal '(#\L #\i #\n #\e #\space #\1 #\0) (vterm-screen-chars scr 9 0 10 80))
  (test-equal 9 (vterm-state-cursor-row st))
  (test-equal 7 (vterm-state-cursor-col st))

  (vterm-screen-resize scr 20 80)
  (test-equal '(#\T #\o #\p) (vterm-screen-chars scr 0 0 1 80))
  (test-equal '(#\L #\i #\n #\e #\space #\1 #\0) (vterm-screen-chars scr 9 0 10 80))
  (test-equal 9 (vterm-state-cursor-row st))
  (test-equal 7 (vterm-state-cursor-col st))

  ;; Resize shorter with content must scroll
  (reset-state st)
  (vterm-screen-resize scr 25 80)

  (vterm-state-process st "Top\x1b;[25HLine 25\x1b;[15H")
  (test-equal '(#\T #\o #\p) (vterm-screen-chars scr 0 0 1 80))
  (test-equal '(#\L #\i #\n #\e #\space #\2 #\5) (vterm-screen-chars scr 24 0 25 80))
  (test-equal 14 (vterm-state-cursor-row st))
  (test-equal 0 (vterm-state-cursor-col st))

  (vterm-screen-scroll-buffer-enable! scr #t)

  (vterm-screen-resize scr 20 80)
  (test-equal screen-events '((push-line (#\T #\o #\p))
                              (push-line ())      
                              (push-line ())
                              (push-line ())
                              (push-line ())))
  (test-equal '() (vterm-screen-chars scr 0 0 1 80))
  (test-equal '(#\L #\i #\n #\e #\space #\2 #\5) (vterm-screen-chars scr 19 0 20 80))
  (test-equal 9 (vterm-state-cursor-row st))
  (test-equal 0 (vterm-state-cursor-col st))

  ;; Resize taller attempts to pop scrollback
  (set! scr (make-test-vterm-screen st '(pop-line)))
  (reset-state st)
  (vterm-screen-resize scr 25 80)
  (vterm-screen-scroll-buffer-enable! scr #t)

  (vterm-state-process st "Line 1\x1b;[25HBottom\x1b;[15H")
  (test-equal '(#\L #\i #\n #\e #\space #\1) (vterm-screen-chars scr 0 0 1 80))
  (test-equal '(#\B #\o #\t #\t #\o #\m) (vterm-screen-chars scr 24 0 25 80))
  (test-equal 14 (vterm-state-cursor-row st))
  (test-equal 0 (vterm-state-cursor-col st))

  (reset-events)
  (vterm-screen-resize scr 30 80)
  (test-equal screen-events '((pop-line 80)
                              (pop-line 80)      
                              (pop-line 80)
                              (pop-line 80)
                              (pop-line 80)))
  (test-equal '(#\A #\B #\C #\D #\E) (vterm-screen-chars scr 0 0 1 80))
  (test-equal '(#\L #\i #\n #\e #\space #\1) (vterm-screen-chars scr 5 0 6 80))
  (test-equal '(#\B #\o #\t #\t #\o #\m) (vterm-screen-chars scr 29 0 30 80))
)
(test-end)

(exit (if (zero? (test-runner-fail-count (test-runner-get))) 0 1))
