;; -*- mode: scheme; coding: utf-8 -*-
;; Copyright (c) 2023 
;; SPDX-License-Identifier: MIT
#!r6rs

(library (vterm)
  (export
    vterm-pos?
    vterm-pos=?
    vterm-pos-row
    vterm-pos-col

    vterm-rect?
    vterm-rect-start-row
    vterm-rect-start-col
    vterm-rect-end-row
    vterm-rect-end-col

    make-vterm-cell
    vterm-cell?
    vterm-cell=?
    vterm-cell-char
    vterm-cell-char-set!
    vterm-cell-width
    vterm-cell-width-set!
    vterm-cell-style
    vterm-cell-style-set!
    vterm-cell-set!

    make-vterm
    make-vterm/screen
    vterm-screen
    vterm?
    vterm-reset
    vterm-process
    vterm-output-port
    vterm-output-port-set!
    vterm-rows
    vterm-cols
    vterm-pen
    vterm-cursor-row
    vterm-cursor-col
    vterm-default-fg
    vterm-default-bg
    vterm-write-key
    vterm-write-char
    vterm-write-string
    vterm-chars
    vterm-string
    vterm-for-each-cell
    vterm-char-ref
    vterm-style-ref
    vterm-resize

    vterm-scroll-buffer-enable!
    vterm-scroll-buffer-enabled?

    vterm-enable-unhandled-seq)
  (import (rnrs)
          (vterm common)
          (vterm screen)
          (vterm state)
          (text-mode console events)
          (only (vterm parser) sexp->vterm-seq))

(define-record-type (vterm $make-vterm vterm?)
  (fields state screen (mutable output-port)))

(define ($make-vterm-handler handler)
  (lambda (sender evt)
    (when handler (handler sender evt))))

(define make-vterm
  (case-lambda
    [()
     (make-vterm 25 80) ]

    [(rows cols)
     (make-vterm rows cols #f) ]

    [(rows cols handler)
     (let ([state (make-vterm-state
                    rows cols ($make-vterm-handler handler))])
       (vterm-state-utf8-enable! state #t)
       ($make-vterm state #f #f)) ]
  ))

(define make-vterm/screen
  (case-lambda
    [()
     (make-vterm/screen 25 80) ]

    [(rows cols)
     (make-vterm/screen rows cols #f) ]

    [(rows cols handler)
     (let* ([state (make-vterm-state rows cols)]
            [scr (make-vterm-screen state ($make-vterm-handler handler))])
       (vterm-state-utf8-enable! state #t)
       ($make-vterm state scr #f)) ]
  ))

(define (vterm-reset vt)
  (vterm-state-reset (vterm-state vt)))

(define (vterm-rows vt)
  (vterm-state-rows (vterm-state vt)))

(define (vterm-cols vt)
  (vterm-state-cols (vterm-state vt)))

(define (vterm-cursor-row vt)
  (vterm-state-cursor-row (vterm-state vt)))

(define (vterm-cursor-col vt)
  (vterm-state-cursor-col (vterm-state vt)))

(define (vterm-process vt bv)
  (vterm-state-process (vterm-state vt) bv))

(define (vterm-pen vt)
  (vterm-state-pen (vterm-state vt)))

(define (vterm-default-fg vt)
  (vterm-state-default-fg (vterm-state vt)))

(define (vterm-default-bg vt)
  (vterm-state-default-bg (vterm-state vt)))

(define (modifier-mask mods)
  (fx+ 1 (bitwise-ior
           (if (enum-set-member? 'shift mods) 1 0)
           (if (enum-set-member? 'alt mods)   2 0)
           (if (enum-set-member? 'ctrl mods)  4 0))) )

(define vterm-write-char
  (case-lambda
    [(vt ch)
     (vterm-write-char vt ch (modifier-set)) ]

    [(vt ch mods)
     (assert (vterm-output-port vt))
     (let ([mods (if (not (eqv? ch #\space))
                      (enum-set-difference mods (modifier-set shift))
                    mods)])
       (if (enum-set=? mods (modifier-set))
           (put-bytevector (vterm-output-port vt) (string->utf8 (list->string (list ch))))
         (let* ([needs-csi-u? (case ch
                                ((#\i #\j #\m #\[) #t)
                                ((#\\ #\] #\^ #\_) #f)
                                ((#\space) (enum-set-member? 'shift mods))
                                (else (or (fx<? (char->integer ch) (char->integer #\a))
                                          (fx>? (char->integer ch) (char->integer #\z))))
                              )]
                [p (vterm-output-port vt)])
           (cond
             ((and needs-csi-u? (not (enum-set=? (modifier-set alt) mods)))
              (put-bytevector p
                (string->utf8
                  (sexp->vterm-seq `((csi #\u (,(char->integer ch) ,(modifier-mask mods)) ))))) )
             (else
               (when (enum-set-member? 'alt mods)
                 (put-u8 p #x1b))
               (put-u8 p (if (enum-set-member? 'ctrl mods)
                             (bitwise-and #x1f (char->integer ch))
                           (char->integer ch))) )
             ))))
     (flush-output-port (vterm-output-port vt))
    ]
))

(define key-codes
 ;;Name       type        literal csi-num class
 '((Enter       enter       #\return #f norm)
   (Tab         tab         #\tab    #f norm)
   (Backspace   literal     #\delete #f norm)
   (Escape      literal     #\esc    #f norm)
   (ArrowUp     csi-cursor  #\A      #f norm)
   (ArrowDown   csi-cursor  #\B      #f norm)
   (ArrowLeft   csi-cursor  #\D      #f norm)
   (ArrowRight  csi-cursor  #\C      #f norm)
   (Insert      csi-num     #\~      2  norm)
   (Delete      csi-num     #\~      3  norm)
   (Home        csi-cursor  #\H      #f norm)
   (End         csi-cursor  #\F      #f norm)
   (PageUp      csi-num     #\~      5  norm)
   (PageDown    csi-num     #\~      6  norm)
   (F1          ss3         #\P     #f  func)
   (F2          ss3         #\Q     #f  func)
   (F3          ss3         #\R     #f  func)
   (F4          ss3         #\S     #f  func)
   (F5          cis-num     #\~     15  func)
   (F6          cis-num     #\~     17  func)
   (F7          cis-num     #\~     18  func)
   (F8          cis-num     #\~     19  func)
   (F9          cis-num     #\~     20  func)
   (F10         cis-num     #\~     21  func)
   (F11         cis-num     #\~     23  func)
   (F12         cis-num     #\~     24  func)
))

(define (get-key key)
  (let ([m (assq key key-codes)])
    (if m
        (values (list-ref m 0) (list-ref m 1) (list-ref m 2) (list-ref m 3) (list-ref m 4))
        (values #f #f #f #f #f))))

(define (write-key-literal vt key literal mods)
  (let ([p (vterm-output-port vt)])
    (if (or (enum-set-member? 'shift mods) (enum-set-member? 'ctrl mods))
        (put-bytevector p
          (string->utf8
            (sexp->vterm-seq `((csi #\u (,(char->integer literal) ,(modifier-mask mods)) )) )))
      (begin
        (when (enum-set-member? 'alt mods)
          (put-u8 p #x1b))
        (put-u8 p (char->integer literal))))))

(define (write-key-csi vt key literal mods)
  (let ([p (vterm-output-port vt)])
    (if (enum-set=? mods (modifier-set))
        (put-bytevector p
          (string->utf8
            (sexp->vterm-seq `((csi ,literal )) )))
        (put-bytevector p
          (string->utf8
            (sexp->vterm-seq `((csi ,literal (1 ,(modifier-mask mods)) )) )))
)))

(define (write-key-csi-num vt key literal num mods)
  (let ([p (vterm-output-port vt)])
    (if (enum-set=? mods (modifier-set))
        (put-bytevector p
          (string->utf8
            (sexp->vterm-seq `((csi ,literal (,num) )) )))
        (put-bytevector p
          (string->utf8
            (sexp->vterm-seq `((csi ,literal (,num ,(modifier-mask mods)) )) )))
)))

(define (write-key-ss3 vt key literal mods)
  (let ([p (vterm-output-port vt)])
    (if (enum-set=? mods (modifier-set))
        (put-bytevector p
          (string->utf8
            (sexp->vterm-seq `((esc ,(list->string (list #\O literal)) )) )))
        (write-key-csi vt key literal mods))))

(define vterm-write-key
  (case-lambda
    [(vt key)
     (vterm-write-key vt key (modifier-set)) ]

    [(vt key mods)
     (let-values ([(key type literal num class) (get-key key)]
                  [(p) (values (vterm-output-port vt))])
       (when key
         (case type
           ((tab)
            (cond
              ((enum-set=? mods (modifier-set shift))
               (put-bytevector p
                 (string->utf8
                   (sexp->vterm-seq `((csi #\Z )) ))) )
              ((enum-set-member? 'shift mods)
               (put-bytevector p
                 (string->utf8
                   (sexp->vterm-seq `((csi #\Z (1 ,(modifier-mask mods)) )) ))) )
              (else
                (write-key-literal vt key literal mods)) ))

           ((enter)
            (if (vterm-state-has-mode? (vterm-state vt) 'newline)
                (put-bytevector p (string->utf8 "\r\n"))
                (write-key-literal vt key literal mods)) )

           ((literal)
            (write-key-literal vt key literal mods) )

           ((csi-num)
            (write-key-csi-num vt key literal num mods))

           ((ss3)
            (write-key-ss3 vt key literal mods))

           ((csi-cursor)
            (if (vterm-state-has-mode? (vterm-state vt) 'cursor)
                (write-key-ss3 vt key literal mods)
                (write-key-csi vt key literal mods)))
         )
       (flush-output-port p))
     )]
))

(define (vterm-write-string vt str)
  (assert (vterm-output-port vt))
  (put-bytevector (vterm-output-port vt) (string->utf8 str))
  (flush-output-port (vterm-output-port vt)))

(define vterm-chars
  (case-lambda
    [(vt)
     (vterm-chars vt 0 0 (vterm-rows vt) (vterm-cols vt)) ]

    [(vt start-row start-col end-row end-col)
     (assert (vterm-screen vt))
     (vterm-screen-chars (vterm-screen vt) start-row start-col end-row end-col)]
  ))

(define vterm-string
  (case-lambda
    [(vt)
     (vterm-string vt 0 0 (vterm-rows vt) (vterm-cols vt)) ]

    [(vt start-row start-col end-row end-col)
     (assert (vterm-screen vt))
     (vterm-screen-string (vterm-screen vt) start-row start-col end-row end-col)]
  ))

(define (vterm-char-ref vt row col)
  (assert (vterm-screen vt))
  (let ([cell (vterm-screen-ref (vterm-screen vt) row col)])
    (and cell (vterm-cell-char cell)) ))

(define (vterm-style-ref vt row col)
  (let ([cell (vterm-screen-ref (vterm-screen vt) row col)])
    (and cell (vterm-cell-style cell)) ))

(define vterm-for-each-cell
  (case-lambda
    [(vt func)
     (vterm-for-each-cell vt 0 0 (vterm-rows vt) (vterm-cols vt) func) ]

    [(vt rect func)
     (vterm-for-each-cell vt (vterm-rect-start-row rect) (vterm-rect-start-col rect)
                                (vterm-rect-end-row rect) (vterm-rect-end-col rect)
                                func) ]

    [(vt start-row start-col end-row end-col func)
     (assert (vterm-screen vt))
     (vterm-screen-for-each (vterm-screen vt) start-row start-col end-row end-col
       (lambda (scr cell row col)
         (func cell row col)))]
))

(define (vterm-enable-unhandled-seq vt enabled?)
  (vterm-state-enable-unhandled-seq-set! (vterm-state vt) enabled?))

(define (vterm-resize vt rows cols)
  (vterm-state-resize (vterm-state vt) rows cols))

(define (vterm-scroll-buffer-enable! vt enable?)
  (assert (vterm-screen vt))
  (vterm-screen-scroll-buffer-enable! (vterm-screen vt) enable?))

(define (vterm-scroll-buffer-enabled? vt)
  (assert (vterm-screen vt))
  (vterm-screen-scroll-buffer-enabled? (vterm-screen vt)))

)
